from django.conf import settings
from django.core import mail
from django.test import override_settings

from kitsune.products.tests import ProductFactory
from kitsune.sumo.tests import TestCase, post
from kitsune.sumo.urlresolvers import reverse
from kitsune.users.tests import GroupFactory, UserFactory, add_permission
from kitsune.wiki.config import (
    MAJOR_SIGNIFICANCE,
    MEDIUM_SIGNIFICANCE,
    SIGNIFICANCES,
    TYPO_SIGNIFICANCE,
)
from kitsune.wiki.events import (
    ApproveRevisionInLocaleEvent,
    ReadyRevisionEvent,
    ReviewableRevisionInLocaleEvent,
)
from kitsune.wiki.models import Revision
from kitsune.wiki.tests import (
    ApprovedRevisionFactory,
    DocumentFactory,
    RevisionFactory,
    new_document_data,
)


def _assert_ready_mail(mail):
    assert "ready for localization" in mail.subject


def _assert_approved_mail(mail):
    assert "new approved revision" in mail.subject


def _assert_creator_mail(mail):
    assert mail.subject.startswith("Your revision has been approved")


def _set_up_ready_watcher():
    """Make a user who watches for revision readiness."""
    ready_watcher = UserFactory(email="ready@example.com")
    ReadyRevisionEvent.notify(ready_watcher)
    return ready_watcher


@override_settings(CELERY_TASK_ALWAYS_EAGER=True)
class ReviewTests(TestCase):
    """Tests for notifications sent during revision review"""

    def setUp(self):
        """Have a user watch for revision approval. Log in."""
        self.group = GroupFactory()
        self.approved_watcher = UserFactory(email="approved@example.com", groups=[self.group])
        ApproveRevisionInLocaleEvent.notify(self.approved_watcher, locale="en-US")
        approver = UserFactory(groups=[self.group])
        add_permission(approver, Revision, "review_revision")
        add_permission(approver, Revision, "mark_ready_for_l10n")
        self.client.login(username=approver.username, password="testpass")

    def _review_revision(
        self,
        is_approved=True,
        is_ready=False,
        significance=SIGNIFICANCES[0][0],
        r=None,
        comment=None,
        **kwargs,
    ):
        """Make a revision, and approve or reject it through the view."""
        if not r:
            r = RevisionFactory(
                is_approved=False,
                is_ready_for_localization=False,
                significance=significance,
                **kwargs,
            )

        # Figure out POST data:
        data = {"comment": "đSome comment"}
        if is_approved:
            data["approve"] = "Approve Revision"
            data["significance"] = significance
            if is_ready:
                data["is_ready_for_localization"] = "on"
            if comment:
                data["comment"] = comment
        else:
            data["reject"] = "Reject Revision"

        url = reverse(
            "wiki.review_revision", locale=r.document.locale, args=[r.document.slug, r.id]
        )
        response = self.client.post(url, data)

        self.assertEqual(302, response.status_code)

    def test_ready(self):
        """Show that a ready(-and-approved) rev mails Ready watchers a Ready
        notification and Approved watchers an Approved one."""
        _set_up_ready_watcher()
        self._review_revision(is_ready=True, significance=MEDIUM_SIGNIFICANCE)
        # 1 mail to each watcher and 1 to the creator
        self.assertEqual(3, len(mail.outbox))
        _assert_ready_mail(mail.outbox[0])
        _assert_approved_mail(mail.outbox[1])
        _assert_creator_mail(mail.outbox[2])

    def test_when_restricted(self):
        """
        Test notifications for ReadyRevisionEvent and ApproveRevisionInLocaleEvent
        when the revision is for a restricted document.
        """
        rw = _set_up_ready_watcher()
        creator = UserFactory(groups=[GroupFactory(name=settings.STAFF_GROUP)])
        self._review_revision(
            is_ready=True,
            creator=creator,
            significance=MEDIUM_SIGNIFICANCE,
            document__restrict_to_groups=[self.group],
        )
        # 1 mail to the approved watcher and 1 to the creator, but
        # none to the ready watcher, since that user is not a member of self.group.
        self.assertEqual(2, len(mail.outbox))
        _assert_approved_mail(mail.outbox[0])
        _assert_creator_mail(mail.outbox[1])

        mail.outbox = []
        other_group = GroupFactory()
        rw.groups.add(other_group)
        reviewer = UserFactory(is_superuser=True)
        self.client.login(username=reviewer.username, password="testpass")
        self._review_revision(
            is_ready=True,
            creator=creator,
            significance=MEDIUM_SIGNIFICANCE,
            document__restrict_to_groups=[other_group],
        )
        # 1 mail to the ready watcher and 1 to the creator, but none
        # to the approved watcher, since that user is not a member of the "other" group.
        self.assertEqual(2, len(mail.outbox))
        _assert_ready_mail(mail.outbox[0])
        _assert_creator_mail(mail.outbox[1])

    def test_product_specific_ready(self):
        """Verify product-specific ready for review notifications."""
        # Add an all products in 'es' watcher and a Firefox OS in 'es'
        # watcher.
        ApproveRevisionInLocaleEvent.notify(UserFactory(), locale="es")
        ApproveRevisionInLocaleEvent.notify(UserFactory(), product="firefox-os", locale="es")

        # Create an 'es' document for Firefox
        parent = DocumentFactory()
        doc = DocumentFactory(parent=parent, locale="es")
        parent.products.add(ProductFactory(slug="firefox"))

        # Review a revision. There should be 2 new emails:
        # 1 to the creator and 1 to the 'es' watcher.
        self._review_revision(document=doc, is_ready=True, significance=MEDIUM_SIGNIFICANCE)
        self.assertEqual(2, len(mail.outbox))
        _assert_approved_mail(mail.outbox[0])
        _assert_creator_mail(mail.outbox[1])

        # Add firefox-os to the document's products and review a new revision.
        # There should be 3 new emails now (the same 2 from before plus one
        # for the firefox-os watcher).
        parent.products.add(ProductFactory(slug="firefox-os"))
        self._review_revision(document=doc, is_ready=True, significance=MEDIUM_SIGNIFICANCE)
        self.assertEqual(5, len(mail.outbox))
        _assert_approved_mail(mail.outbox[2])
        _assert_approved_mail(mail.outbox[3])
        _assert_creator_mail(mail.outbox[4])

        # Add a Firefox watcher. This time there should be 4 new emails.
        ApproveRevisionInLocaleEvent.notify(UserFactory(), product="firefox", locale="es")
        self._review_revision(document=doc, is_ready=True, significance=MEDIUM_SIGNIFICANCE)
        self.assertEqual(9, len(mail.outbox))

    def test_typo_significance_ignore(self):
        # Create the first approved revision for the document. This one will
        # always have MAJOR_SIGNIFICANCE.
        r = RevisionFactory(is_approved=True)

        # Then, set up a watcher and create a TYPO_SIGNIFICANCE revision.
        _set_up_ready_watcher()
        self._review_revision(is_ready=True, document=r.document, significance=TYPO_SIGNIFICANCE)
        # This is the same as test_ready, except we miss 1 mail, that is the
        # localization mail.
        self.assertEqual(2, len(mail.outbox))

    def test_approved(self):
        """Show that an approved rev mails Ready watchers nothing and Approved
        watchers an Approved notification."""
        _set_up_ready_watcher()
        self._review_revision(is_ready=False)
        # 1 mail to Approved watcher and 1 to creator
        self.assertEqual(2, len(mail.outbox))
        assert "new approved revision" in mail.outbox[0].subject
        assert "Your revision has been approved" in mail.outbox[1].subject

    def test_based_on_approved(self):
        u1 = UserFactory()
        r1 = RevisionFactory(is_approved=False, creator=u1, is_ready_for_localization=False)
        u2 = UserFactory()
        r2 = RevisionFactory(
            document=r1.document,
            based_on=r1,
            is_approved=False,
            creator=u2,
            is_ready_for_localization=False,
        )
        self.assertEqual(0, len(mail.outbox))
        self._review_revision(r=r2)
        # 1 mail for the watcher and 1 for creator.
        self.assertEqual(3, len(mail.outbox))
        assert "has a new approved revision" in mail.outbox[0].subject
        assert "Your revision has been approved" in mail.outbox[1].subject
        assert "A revision you contributed to has" in mail.outbox[2].subject

    def test_neither(self):
        """Show that neither an Approved nor a Ready mail is sent if a rev is
        rejected."""
        _set_up_ready_watcher()
        self._review_revision(is_approved=False)
        self.assertEqual(1, len(mail.outbox))  # 1 mail to creator.
        assert mail.outbox[0].subject.startswith("Your revision has been reviewed")

    def test_user_watching_both(self):
        """If a single person is watching ready and approved revisions and a
        revision becomes ready, send only the readiness email, not the approval
        one."""
        # Have the Approved watcher watch Ready as well:
        ReadyRevisionEvent.notify(self.approved_watcher)

        self._review_revision(is_ready=True, significance=MEDIUM_SIGNIFICANCE)
        # 1 mail to watcher and 1 to creator
        self.assertEqual(2, len(mail.outbox))
        _assert_ready_mail(mail.outbox[0])
        _assert_creator_mail(mail.outbox[1])

    def test_new_lines_in_review_message(self):
        """Test that newlines in a review message are properly displayed."""
        _set_up_ready_watcher()
        self._review_revision(comment="foo\n\nbar\nbaz")
        msg = mail.outbox[1].alternatives[0][0]
        assert "foo</p>" in msg
        assert "bar<br>baz</p>" in msg

    def test_first_approved_revision_has_major_significance(self):
        """The 1st approved revision of a document has MAJOR_SIGNIFICANCE."""
        self._review_revision(significance=MEDIUM_SIGNIFICANCE)
        r = Revision.objects.get()

        # Even though MEDIUM_SIGNIFICANCE was POSTed, the revision will be set
        # to MAJOR_SIGNIFICANCE.
        self.assertEqual(MAJOR_SIGNIFICANCE, r.significance)


@override_settings(CELERY_TASK_ALWAYS_EAGER=True)
class ReadyForL10nTests(TestCase):
    """Tests for notifications sent during ready for l10n"""

    def setUp(self):
        """Have a user watch for revision approval. Log in."""
        self.ready_watcher = UserFactory(email="approved@example.com")
        ReadyRevisionEvent.notify(self.ready_watcher)

        readyer = UserFactory()
        add_permission(readyer, Revision, "mark_ready_for_l10n")
        self.client.login(username=readyer.username, password="testpass")

    def _mark_as_ready_revision(self, doc=None):
        """Make a revision, and approve or reject it through the view."""
        if doc is None:
            doc = DocumentFactory()

        r = ApprovedRevisionFactory(
            is_ready_for_localization=False, significance=MEDIUM_SIGNIFICANCE, document=doc
        )

        # Figure out POST data:
        data = {"comment": "something"}

        response = post(
            self.client, "wiki.mark_ready_for_l10n_revision", data, args=[r.document.slug, r.id]
        )
        self.assertEqual(200, response.status_code)

    def test_ready(self):
        """Show that a ready(-and-approved) rev mails Ready watchers a Ready
        notification and Approved watchers an Approved one."""
        _set_up_ready_watcher()
        self._mark_as_ready_revision()
        self.assertEqual(2, len(mail.outbox))  # 1 mail to each watcher, none to marker
        _assert_ready_mail(mail.outbox[0])
        _assert_ready_mail(mail.outbox[1])

    def test_product_specific_ready(self):
        """Verify product-specific ready for l10n notifications."""
        # Add a Firefox OS watcher.
        ReadyRevisionEvent.notify(UserFactory(), product="firefox-os")

        # Create a document for Firefox
        doc = DocumentFactory()
        doc.products.add(ProductFactory(slug="firefox"))

        # Mark a revision a ready for L10n. There should be only one email
        # to the watcher created in setUp.
        self._mark_as_ready_revision(doc=doc)
        self.assertEqual(1, len(mail.outbox))
        _assert_ready_mail(mail.outbox[0])

        # Add firefox-os to the document's products. Mark as ready for l10n,
        # and there should be two new emails.
        doc.products.add(ProductFactory(slug="firefox-os"))
        self._mark_as_ready_revision(doc=doc)
        self.assertEqual(3, len(mail.outbox))
        _assert_ready_mail(mail.outbox[1])
        _assert_ready_mail(mail.outbox[2])

        # Add a Firefox watcher, mark as ready for l10n, and there should
        # be three new emails.
        ReadyRevisionEvent.notify(UserFactory(), product="firefox")
        self._mark_as_ready_revision(doc=doc)
        self.assertEqual(6, len(mail.outbox))


@override_settings(CELERY_TASK_ALWAYS_EAGER=True)
class ReviewableRevisionInLocaleEventTests(TestCase):
    """Tests notifications when a revision is created."""

    def setUp(self):
        self.group1 = GroupFactory()
        self.group2 = GroupFactory()
        self.group3 = GroupFactory()
        self.watcher1 = UserFactory(email="ringo@example.com", groups=[self.group1, self.group2])
        self.watcher2 = UserFactory(email="mccartney@example.com", groups=[self.group3])
        add_permission(self.watcher1, Revision, "review_revision")
        add_permission(self.watcher2, Revision, "review_revision")
        ReviewableRevisionInLocaleEvent.notify(self.watcher1, locale="en-US")
        ReviewableRevisionInLocaleEvent.notify(self.watcher2, locale="en-US")
        creator = UserFactory(groups=[GroupFactory(name=settings.STAFF_GROUP)])
        self.client.login(username=creator.username, password="testpass")

    def test_when_restricted(self):
        """
        Test notifications for ReviewableRevisionInLocaleEvent when creating
        a revision for a restricted document.
        """
        data = new_document_data()
        data["restrict_to_groups"] = [self.group1.id, self.group2.id]
        response = self.client.post(reverse("wiki.new_document"), data)
        self.assertEqual(302, response.status_code)

        self.assertEqual(1, len(mail.outbox))
        self.assertIn(f"{data['title']} is ready for review", mail.outbox[0].subject)
        self.assertIn(self.watcher1.email, mail.outbox[0].to)

        mail.outbox = []

        data["title"] = "Another Test Article"
        data["slug"] = "another-test-article"
        data["restrict_to_groups"] = [self.group3.id]
        response = self.client.post(reverse("wiki.new_document"), data)
        self.assertEqual(302, response.status_code)

        self.assertEqual(1, len(mail.outbox))
        self.assertIn(f"{data['title']} is ready for review", mail.outbox[0].subject)
        self.assertIn(self.watcher2.email, mail.outbox[0].to)
