import logging

import requests
from django.conf import settings
from django.contrib import messages
from django.contrib.auth import get_user_model
from django.contrib.auth.models import User
from django.db import transaction
from django.urls import reverse as django_reverse
from django.utils.translation import activate
from django.utils.translation import gettext as _
from mozilla_django_oidc.auth import OIDCAuthenticationBackend

from kitsune.customercare.tasks import update_zendesk_identity
from kitsune.products.models import Product
from kitsune.sumo.urlresolvers import reverse
from kitsune.users.models import Profile
from kitsune.users.utils import add_to_contributors, get_oidc_fxa_setting

log = logging.getLogger("k.users")


def is_mozilla_domain_email(email: str) -> bool:
    """Check if the email domain is in the MOZILLA_DOMAINS list."""
    if not email:
        return False

    domain = email.split('@')[-1].lower()
    return domain in [d.lower() for d in settings.MOZILLA_DOMAINS]


class SumoOIDCAuthBackend(OIDCAuthenticationBackend):
    def authenticate(self, request, **kwargs):
        """Authenticate a user based on the OIDC code flow."""

        # If the request has the /fxa/callback/ path then probably there is a login
        # with Mozilla accounts. In this case just return None and let
        # the FxA backend handle this request.
        if request and not request.path == django_reverse("oidc_authentication_callback"):
            return None

        return super().authenticate(request, **kwargs)


class FXAAuthBackend(OIDCAuthenticationBackend):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.refresh_token = None

    @staticmethod
    def get_settings(attr, *args):
        """Override settings for Mozilla accounts Provider."""
        val = get_oidc_fxa_setting(attr)
        if val is not None:
            return val
        return super(FXAAuthBackend, FXAAuthBackend).get_settings(attr, *args)

    def get_token(self, payload):
        token_info = super().get_token(payload)
        self.refresh_token = token_info.get("refresh_token")
        return token_info

    @classmethod
    def refresh_access_token(cls, refresh_token, ttl=None):
        """Gets a new access_token by using a refresh_token.

        returns: the actual token or an empty dictionary
        """

        if not refresh_token:
            return {}

        obj = cls()
        payload = {
            "client_id": obj.OIDC_RP_CLIENT_ID,
            "client_secret": obj.OIDC_RP_CLIENT_SECRET,
            "grant_type": "refresh_token",
            "refresh_token": refresh_token,
        }

        if ttl:
            payload.update({"ttl": ttl})

        try:
            return obj.get_token(payload=payload)
        except requests.exceptions.HTTPError:
            return {}

    def update_contributor_status(self, profile):
        """Register user as contributor."""
        # The request attribute might not be set.
        request = getattr(self, "request", None)
        if request and (contribution_area := request.session.get("contributor")):
            add_to_contributors(profile.user, profile.locale, contribution_area)
            del request.session["contributor"]

    def create_user(self, claims):
        """Override create user method to mark the profile as migrated."""

        user = super().create_user(claims)
        # Create a user profile for the user and populate it with data from
        # Mozilla accounts
        profile, created = Profile.objects.get_or_create(user=user)
        profile.is_fxa_migrated = True
        profile.fxa_uid = claims.get("uid")
        profile.fxa_avatar = claims.get("avatar", "")
        profile.name = claims.get("displayName", "")
        subscriptions = claims.get("subscriptions", [])

        if email := claims.get("email"):
            profile.is_mozilla_staff = is_mozilla_domain_email(email)

        # Let's get the first element even if it's an empty string
        # A few assertions return a locale of None so we need to default to empty string
        fxa_locale = (claims.get("locale", "") or "").split(",")[0]
        if fxa_locale in settings.SUMO_LANGUAGES:
            profile.locale = fxa_locale
        else:
            profile.locale = self.request.session.get("login_locale", settings.LANGUAGE_CODE)
        activate(profile.locale)

        # If there is a refresh token, store it
        if self.refresh_token:
            profile.fxa_refresh_token = self.refresh_token
        profile.save()
        # User subscription information
        products = Product.active.filter(codename__in=subscriptions)
        profile.products.clear()
        profile.products.add(*products)

        # This is a new sumo profile, show edit profile message
        messages.success(
            self.request,
            _(
                "<strong>Welcome!</strong> You are now signed in using your Mozilla account. "
                + "{a_profile}Edit your profile.{a_close}<br>"
                + "Already have a different Mozilla Support Account? "
                + "{a_more}Read more.{a_close}"
            ).format(
                a_profile='<a href="' + reverse("users.edit_my_profile") + '" target="_blank">',
                a_more='<a href="'
                + reverse("wiki.document", args=["firefox-accounts-mozilla-support-faq"])
                + '" target="_blank">',
                a_close="</a>",
            ),
            extra_tags="safe",
        )

        # update contributor status
        self.update_contributor_status(profile)

        return user

    def filter_users_by_claims(self, claims):
        """Match users by FxA uid or email."""
        fxa_uid = claims.get("uid")
        user_model = get_user_model()
        users = user_model.objects.none()

        # something went terribly wrong. Return None
        if not fxa_uid:
            log.warning("Failed to get Mozilla account UID.")
            return users

        # A existing user is attempting to connect a Mozilla account to the SUMO profile
        # NOTE: this section will be dropped when the migration is complete
        if self.request and self.request.user and self.request.user.is_authenticated:
            return [self.request.user]

        users = user_model.objects.filter(profile__fxa_uid=fxa_uid)

        if not users:
            # We did not match any users so far. Let's call the super method
            # which will try to match users based on email
            users = super().filter_users_by_claims(claims)
        return users

    def get_userinfo(self, access_token, id_token, payload):
        """Return user details and subscription information dictionary."""

        user_info = super().get_userinfo(access_token, id_token, payload)

        if not settings.FXA_OP_SUBSCRIPTION_ENDPOINT:
            return user_info

        # Fetch subscription information
        try:
            sub_response = requests.get(
                settings.FXA_OP_SUBSCRIPTION_ENDPOINT,
                headers={"Authorization": "Bearer {}".format(access_token)},
                verify=self.get_settings("OIDC_VERIFY_SSL", True),
            )
            sub_response.raise_for_status()
        except requests.exceptions.RequestException:
            log.error("Failed to fetch subscription status", exc_info=True)
            # if something went wrong, just return whatever the profile endpoint holds
            return user_info
        # This will override whatever the profile endpoint returns
        # until https://github.com/mozilla/fxa/issues/2463 is fixed
        user_info["subscriptions"] = sub_response.json().get("subscriptions", [])
        return user_info

    def update_user(self, user, claims):
        """Update existing user with new claims, if necessary save, and return user"""
        profile = user.profile
        fxa_uid = claims.get("uid")
        email = claims.get("email")
        user_attr_changed = False
        # Check if the user has active subscriptions
        subscriptions = claims.get("subscriptions", [])

        if (request := getattr(self, "request", None)) and not profile.is_fxa_migrated:
            # Check if there is already a Mozilla account with this ID
            if Profile.objects.filter(fxa_uid=fxa_uid).exists():
                msg = _("This Mozilla account is already used in another profile.")
                messages.error(request, msg)
                return None

            # If it's not migrated, we can assume that there isn't an FxA id too
            profile.is_fxa_migrated = True
            profile.fxa_uid = fxa_uid
            # This is the first time an existing user is using FxA. Redirect to profile edit
            # in case the user wants to update any settings.
            request.session["oidc_login_next"] = reverse("users.edit_my_profile")
            messages.info(request, "fxa_notification_updated")

        # There is a change in the email in Mozilla accounts. Let's update user's email
        # unless we have a superuser
        if email and (email != user.email) and not user.profile.in_staff_group:
            if User.objects.exclude(id=user.id).filter(email=email).exists():
                if request:
                    msg = _(
                        "The e-mail address used with this Mozilla account is already "
                        "linked in another profile."
                    )
                    messages.error(request, msg)
                return None
            user.email = email
            user_attr_changed = True

        # Follow avatars from FxA profiles
        profile.fxa_avatar = claims.get("avatar", "")
        # User subscription information
        products = Product.active.filter(codename__in=subscriptions)
        profile.products.clear()
        profile.products.add(*products)

        # update contributor status
        self.update_contributor_status(profile)

        # Users can select their own display name.
        if not profile.name:
            profile.name = claims.get("displayName", "")

        # If there is a refresh token, store it
        if self.refresh_token:
            profile.fxa_refresh_token = self.refresh_token

        profile.is_mozilla_staff = is_mozilla_domain_email(email)

        with transaction.atomic():
            if user_attr_changed:
                user.save()
            profile.save()

        # If we have an updated email, let's update Zendesk too
        # the check is repeated for now but it will save a few
        # API calls if we trigger the task only when we know that we have new emails
        if user_attr_changed:
            update_zendesk_identity.delay(user.id, email)

        return user

    def authenticate(self, request, **kwargs):
        """Authenticate a user based on the OIDC/oauth2 code flow."""

        # If the request has the /oidc/callback/ path then probably there is a login
        # attempt in the admin interface. In this case just return None and let
        # the OIDC backend handle this request.
        if request and request.path == django_reverse("oidc_authentication_callback"):
            return None

        return super().authenticate(request, **kwargs)
