import re

import django.urls
from django.conf import settings
from django.core.exceptions import ObjectDoesNotExist
from django.utils import translation

LANGUAGES_WITH_FALLBACKS_REGEX = re.compile(
    rf"^/(?P<language>{'|'.join(settings.NON_SUPPORTED_LOCALES.keys())})(?P<slash>/|$)",
    re.IGNORECASE,
)
FALLBACK_LANGUAGE_URL_MAP = {
    k.lower(): v if v else "en-US" for k, v in settings.NON_SUPPORTED_LOCALES.items()
}
MAP_TO_TRADITIONAL_CHINESE = frozenset(("zh-hant", "zh-hk", "zh-mo"))


class LocalePrefixPattern(django.urls.LocalePrefixPattern):
    """
    A sub-class of Django's "LocalePrefixPattern" that simply normalizes the language
    prefix for SUMO (e.g., upper-case country codes). This is an essential piece, since
    it controls the resolution of the language code prefix of incoming URL's as well
    as the language code prefix of the URL's generated by Django's "reverse".
    """

    @property
    def language_prefix(self):
        language_code = normalize_language(translation.get_language()) or settings.LANGUAGE_CODE
        if language_code == settings.LANGUAGE_CODE and not self.prefix_default_language:
            return ""
        return f"{language_code}/"


def i18n_patterns(*urls, prefix_default_language=True):
    """
    Similar to Django's i18n_patterns but uses SUMO's "LocalePrefixPattern".
    """
    if not settings.USE_I18N:
        return list(urls)
    return [
        django.urls.URLResolver(
            LocalePrefixPattern(prefix_default_language=prefix_default_language),
            list(urls),
        )
    ]


def split_into_language_and_path(path):
    """
    Given a URL path that starts with a language, returns the
    language and the rest of the path after the language.
    """
    language, slash, rest_of_path = path.lstrip("/").partition("/")
    return (language, f"{slash}{rest_of_path}")


def normalize_language(language):
    """
    Given a language code, returns the language code supported by SUMO in the
    proper case, for example "eN-us" --> "en-US" or "sC" --> "it", or None if
    SUMO doesn't support the language code.
    """
    if not language:
        return None
    lc_language = language.lower()
    return settings.LANGUAGE_URL_MAP.get(lc_language) or FALLBACK_LANGUAGE_URL_MAP.get(lc_language)


def normalize_path(path, force_language=False):
    """
    Normalizes the language code, if any, that starts the path. If "force_language" is
    given, it will replace any existing language code in the path, or prefix the path
    if the path didn't already start with a language code.

    Example when "force_language=False":
        Requested  --> Normalized
            /en-us --> /en-US  (normalization of case)
            /fr-ca --> /fr     (supported variant)
            /en-gb --> /en-US  (supported variant)
            /sc    --> /it     (explicit fallback)
            /ak    --> /en-US  (explicit fallback)

    Example when "force_language=de":
        Requested    --> Normalized
            /path    --> /de/path  (path prefixed)
            /it/path --> /de/path  (language code replaced)
    """
    # Use Django's standard way to check if the path starts with a supported language.
    language_from_path = translation.get_language_from_path(path)

    # Even if Django found that the path didn't start with a supported language, the path
    # may start with one for which SUMO defines a supported fallback language, for example
    # "sc" --> "it".
    if not language_from_path and (mo := LANGUAGES_WITH_FALLBACKS_REGEX.match(path)):
        language_from_path = mo.group("language")

    normalized_language_from_path = normalize_language(language_from_path)

    if normalized_language_from_path:
        actual_language_from_path, remaining_path = split_into_language_and_path(path)
        if force_language:
            return f"/{force_language}{remaining_path}"
        elif actual_language_from_path != normalized_language_from_path:
            if (normalized_language_from_path == "zh-CN") and (
                actual_language_from_path.lower() in MAP_TO_TRADITIONAL_CHINESE
            ):
                # Django will always map any unsupported Traditional Chinese locale, like
                # "zh-hant" or "zh-hk" for example, into Simplified Chinese ("zh-cn") because
                # we're not using "strict" mode, and "zh-cn" is always the first "zh-" match.
                # For these specific cases we should be mapping to "zh-tw" instead.
                normalized_language_from_path = "zh-TW"
            return f"/{normalized_language_from_path}{remaining_path}"
    elif force_language:
        return f"/{force_language}{path}"

    return path


def get_language_from_user(request):
    """
    Return the normalized language code from the following user-related
    sources, if possible, otherwise return None.
        1) Check for a logged-in user's preferred language in their
           profile.
        2) Check for the user's preferred language in their session.
    """
    language = None

    if request.user.is_authenticated:
        try:
            language = request.user.profile.locale
        except ObjectDoesNotExist:
            pass

    if not language:
        language = request.session.get(settings.LANGUAGE_COOKIE_NAME)

    return normalize_language(language)
