from kitsune.questions.managers import get_sumo_bot
from kitsune.questions.models import Answer, Question
from kitsune.questions.tests import AnswerFactory, QuestionFactory
from kitsune.sumo.tests import TestCase
from kitsune.users.models import Profile
from kitsune.users.tests import UserFactory


class QuestionManagerTestCase(TestCase):
    def setUp(self):
        get_sumo_bot.cache_clear()

    def test_spam(self):
        """Verify the spam queryset."""
        q = QuestionFactory()
        self.assertEqual(0, Question.objects.spam().count())

        q.mark_as_spam(Profile.get_sumo_bot())
        self.assertEqual(1, Question.objects.spam().count())

    def test_detected_spam(self):
        """Verify the detected_spam queryset for bot-marked spam."""
        q1 = QuestionFactory()
        q2 = QuestionFactory()

        q1.mark_as_spam(Profile.get_sumo_bot())
        detected_questions = Question.objects.detected_spam().filter(id__in=[q1.id, q2.id])
        self.assertEqual(1, detected_questions.count())
        self.assertEqual(q1.id, detected_questions.first().id)

        human_user = UserFactory()
        q2.mark_as_spam(human_user)

        detected_questions = Question.objects.detected_spam().filter(id__in=[q1.id, q2.id])
        self.assertEqual(1, detected_questions.count())
        self.assertEqual(q1.id, detected_questions.first().id)

    def test_undetected_spam(self):
        """Verify the undetected_spam queryset for manually marked spam."""
        q1 = QuestionFactory()
        q2 = QuestionFactory()

        human_user = UserFactory()
        q1.mark_as_spam(human_user)

        undetected_questions = Question.objects.undetected_spam().filter(id=q1.id)
        self.assertEqual(1, undetected_questions.count())
        self.assertEqual(q1.id, undetected_questions.first().id)

        q2.mark_as_spam(Profile.get_sumo_bot())

        self.assertTrue(Question.objects.undetected_spam().filter(id=q1.id).exists())
        self.assertFalse(Question.objects.undetected_spam().filter(id=q2.id).exists())

        our_questions = Question.objects.undetected_spam().filter(id__in=[q1.id, q2.id])
        self.assertEqual(1, our_questions.count())
        self.assertEqual(q1.id, our_questions.first().id)

    def test_done(self):
        """Verify the done queryset."""
        # Create a question, there shouldn't be any done yet.
        q = QuestionFactory()
        self.assertEqual(0, Question.objects.done().count())

        # Add an answer, there shouldn't be any done yet.
        a = AnswerFactory(question=q)
        self.assertEqual(0, Question.objects.done().count())

        # Make it the solution, there should be one done.
        q.solution = a
        q.save()
        self.assertEqual(1, Question.objects.done().count())

        # Create a locked questions, there should be two done.
        QuestionFactory(is_locked=True)
        self.assertEqual(2, Question.objects.done().count())

    def test_responded(self):
        """Verify the responded queryset."""
        # Create a question, there shouldn't be any responded yet.
        q = QuestionFactory()
        self.assertEqual(0, Question.objects.responded().count())

        # Add an answer, there should be one responded.
        a = AnswerFactory(question=q)
        self.assertEqual(1, Question.objects.responded().count())

        # Add an answer by the creator, there should be none responded.
        a = AnswerFactory(creator=q.creator, question=q)
        self.assertEqual(0, Question.objects.responded().count())

        # Add another answer, there should be one responded.
        a = AnswerFactory(question=q)
        self.assertEqual(1, Question.objects.responded().count())

        # Lock it, there should be none responded.
        q.is_locked = True
        q.save()
        self.assertEqual(0, Question.objects.responded().count())

        # Unlock it and mark solved, there should be none responded.
        q.is_locked = False
        q.solution = a
        q.save()
        self.assertEqual(0, Question.objects.responded().count())

    def test_needs_attention(self):
        """Verify the needs_attention queryset."""
        # Create a question, there should be one needs_attention.
        q = QuestionFactory()
        self.assertEqual(1, Question.objects.needs_attention().count())

        # Add an answer, there should be no needs_attention.
        a = AnswerFactory(question=q)
        self.assertEqual(0, Question.objects.needs_attention().count())

        # Add an answer by the creator, there should be one needs_attention.
        a = AnswerFactory(creator=q.creator, question=q)
        self.assertEqual(1, Question.objects.needs_attention().count())

        # Lock it, there should be none responded.
        q.is_locked = True
        q.save()
        self.assertEqual(0, Question.objects.needs_attention().count())

        # Unlock it and mark solved, there should be none responded.
        q.is_locked = False
        q.solution = a
        q.save()
        self.assertEqual(0, Question.objects.needs_attention().count())

    def test_needs_info(self):
        """Verify the needs_info queryset."""
        q = QuestionFactory()

        # There should be no needs_info questions yet.
        self.assertEqual(0, Question.objects.needs_info().count())

        # Tag a question and there should be one needs_info question.
        q.set_needs_info()
        self.assertEqual(1, Question.objects.needs_info().count())

        # Remove tag and there shouldn't be any anymore.
        q.unset_needs_info()
        self.assertEqual(0, Question.objects.needs_info().count())


class AnswerManagerTestCase(TestCase):
    def test_not_by_asker(self):
        """Verify that only answers by users other than the original asker are returned"""
        q = QuestionFactory()

        # Add an answer by the original asker
        AnswerFactory(question=q, creator=q.creator)
        self.assertEqual(0, Answer.objects.not_by_asker().count())

        # Add an answer by someone else
        AnswerFactory(question=q)
        self.assertEqual(1, Answer.objects.not_by_asker().count())
