import json
from datetime import datetime

import actstream.actions
import django_filters
from django import forms
from django.db.models import Q
from django_filters.rest_framework import DjangoFilterBackend
from rest_framework import pagination, permissions, serializers, status, viewsets
from rest_framework.decorators import action
from rest_framework.response import Response

from kitsune.products.api_utils import TopicField
from kitsune.products.models import Product, Topic
from kitsune.questions.models import (
    AlreadyTakenException,
    Answer,
    AnswerVote,
    InvalidUserException,
    Question,
    QuestionMetaData,
    QuestionVote,
)
from kitsune.sumo.api_utils import (
    DateTimeUTCField,
    GenericAPIException,
    OnlyCreatorEdits,
    OrderingFilter,
    SplitSourceField,
)
from kitsune.sumo.utils import is_ratelimited
from kitsune.tags.models import SumoTag
from kitsune.tags.utils import add_existing_tag
from kitsune.users.api import ProfileFKSerializer
from kitsune.users.models import Profile


def get_or_create_profile(user):
    """
    Get or create a Profile for a User.

    This helper ensures Users without Profiles don't cause serialization failures.
    Follows Django's get_or_create naming pattern.

    Args:
        user: User instance (can be None for optional fields)

    Returns:
        Profile instance if user is provided, None otherwise
    """
    if user is None:
        return None
    profile, _ = Profile.objects.get_or_create(user=user)
    return profile


class QuestionMetaDataSerializer(serializers.ModelSerializer):
    question = serializers.PrimaryKeyRelatedField(
        required=False, write_only=True, queryset=Question.objects.all()
    )

    class Meta:
        model = QuestionMetaData
        fields = ("name", "value", "question")


class QuestionSerializer(serializers.ModelSerializer):
    answers = serializers.PrimaryKeyRelatedField(many=True, read_only=True)
    content = SplitSourceField(read_source="content_parsed", write_source="content")
    created = DateTimeUTCField(read_only=True)
    creator = serializers.SerializerMethodField()
    involved = serializers.SerializerMethodField()
    is_solved = serializers.ReadOnlyField()
    is_taken = serializers.ReadOnlyField()
    metadata = QuestionMetaDataSerializer(source="metadata_set", read_only=True, many=True)
    num_votes = serializers.ReadOnlyField()
    product = serializers.SlugRelatedField(
        required=True, slug_field="slug", queryset=Product.active.all()
    )
    tags = serializers.SerializerMethodField()
    solution = serializers.PrimaryKeyRelatedField(read_only=True)
    solved_by = serializers.SerializerMethodField()
    taken_by = serializers.SerializerMethodField()
    topic = TopicField(required=True, queryset=Topic.active.all())
    updated = DateTimeUTCField(read_only=True)
    updated_by = serializers.SerializerMethodField()

    class Meta:
        model = Question
        fields = (
            "answers",
            "content",
            "created",
            "creator",
            "id",
            "involved",
            "is_archived",
            "is_locked",
            "is_solved",
            "is_spam",
            "is_taken",
            "last_answer",
            "locale",
            "metadata",
            "tags",
            "num_answers",
            "num_votes_past_week",
            "num_votes",
            "product",
            "solution",
            "solved_by",
            "taken_until",
            "taken_by",
            "title",
            "topic",
            "updated_by",
            "updated",
        )

    def get_involved(self, obj):
        involved_profiles = []
        creator_profile = get_or_create_profile(obj.creator)
        involved_profiles.append(creator_profile)

        for answer in obj.answers.all():
            answer_profile = get_or_create_profile(answer.creator)
            if answer_profile not in involved_profiles:
                involved_profiles.append(answer_profile)

        return ProfileFKSerializer(involved_profiles, many=True).data

    def get_solved_by(self, obj):
        if not obj.solution:
            return None
        profile = get_or_create_profile(obj.solution.creator)
        return ProfileFKSerializer(profile).data

    def get_creator(self, obj):
        profile = get_or_create_profile(obj.creator)
        return ProfileFKSerializer(profile).data

    def get_taken_by(self, obj):
        profile = get_or_create_profile(obj.taken_by)
        return ProfileFKSerializer(profile).data if profile else None

    def get_updated_by(self, obj):
        profile = get_or_create_profile(obj.updated_by)
        return ProfileFKSerializer(profile).data if profile else None

    def get_tags(self, obj):
        return [{"name": tag.name, "slug": tag.slug} for tag in obj.tags.all()]

    def validate(self, data):
        user = self.context.get("request").user
        if user and not user.is_anonymous and data.get("creator") is None:
            data["creator"] = user
        return data


class QuestionFKSerializer(QuestionSerializer):
    class Meta:
        model = Question
        fields = (
            "creator",
            "id",
            "title",
        )


class QuestionFilter(django_filters.FilterSet):
    product = django_filters.CharFilter(field_name="product__slug")
    creator = django_filters.CharFilter(field_name="creator__username")
    involved = django_filters.CharFilter(method="filter_involved")
    is_solved = django_filters.BooleanFilter(method="filter_is_solved", widget=forms.TextInput)
    is_taken = django_filters.BooleanFilter(method="filter_is_taken", widget=forms.TextInput)
    metadata = django_filters.CharFilter(method="filter_metadata")
    solved_by = django_filters.CharFilter(method="filter_solved_by")
    taken_by = django_filters.CharFilter(field_name="taken_by__username")
    updated = django_filters.IsoDateTimeFilter()
    created = django_filters.IsoDateTimeFilter()

    class Meta:
        model = Question
        fields = {
            "creator": ["exact"],
            "created": ["gt", "lt", "exact"],
            "is_archived": ["exact"],
            "is_locked": ["exact"],
            "is_spam": ["exact"],
            "locale": ["exact"],
            "num_answers": ["exact"],
            "product": ["exact"],
            "taken_by": ["exact"],
            "title": ["exact"],
            "topic": ["exact"],
            "updated": ["gt", "lt", "exact"],
            "updated_by": ["exact"],
        }

    def filter_involved(self, queryset, name, value):
        # This will remain unevaluated, and become a subquery of the final query.
        # Using a subquery instead of a JOIN like Django would normally do
        # should be faster in this case.
        questions_user_answered = Answer.objects.filter(creator__username=value).values(
            "question_id"
        )

        answered_filter = Q(id__in=questions_user_answered)
        creator_filter = Q(creator__username=value)
        return queryset.filter(creator_filter | answered_filter)

    def filter_is_taken(self, queryset, name, value):
        now = datetime.now()
        if value:
            # only taken questions
            return queryset.filter(~Q(taken_by=None), taken_until__gt=now)
        # only not taken questions
        return queryset.filter(Q(taken_by=None) | Q(taken_until__lt=now))

    def filter_is_solved(self, queryset, name, value):
        solved_filter = Q(solution=None)
        if value:
            solved_filter = ~solved_filter
        return queryset.filter(solved_filter)

    def filter_solved_by(self, queryset, name, value):
        question_user_solved = Question.objects.filter(solution__creator__username=value).values(
            "id"
        )

        return queryset.filter(id__in=question_user_solved)

    def filter_metadata(self, queryset, name, value):
        try:
            value = json.loads(value)
        except ValueError:
            raise GenericAPIException(status.HTTP_400_BAD_REQUEST, "metadata must be valid JSON.")

        for key, values in list(value.items()):
            if not isinstance(values, list):
                values = [values]
            query = Q()
            for v in values:
                if v is None:
                    query = query | ~Q(metadata_set__name=key)
                else:
                    query = query | Q(metadata_set__name=key, metadata_set__value=v)
            queryset = queryset.filter(query)

        return queryset


class HasRemoveTagPermissions(permissions.BasePermission):
    def has_object_permission(self, request, view, obj):
        """Simple permision check to match the one from the question view."""

        if not request.user.has_perm("questions.remove_tag"):
            return False
        return super().has_object_permission(request, view, obj)


class HasAddTagPermissions(permissions.BasePermission):
    def has_object_permission(self, request, view, obj):
        """Simple permision check to match the one from the question view."""

        if not request.user.has_perm("questions.tag_question"):
            return False
        return super().has_object_permission(request, view, obj)


class QuestionViewSet(viewsets.ModelViewSet):
    serializer_class = QuestionSerializer
    queryset = Question.objects.all()
    pagination_class = pagination.PageNumberPagination
    permission_classes = [
        OnlyCreatorEdits,
        permissions.IsAuthenticatedOrReadOnly,
    ]
    filterset_class = QuestionFilter
    filter_backends = [
        DjangoFilterBackend,
        OrderingFilter,
    ]
    ordering_fields = [
        "id",
        "created",
        "last_answer",
        "num_answers",
        "num_votes_past_week",
        "updated",
    ]
    # Default, if not overwritten
    ordering = ("-id",)

    @action(detail=True, methods=["post"])
    def solve(self, request, pk=None):
        """Accept an answer as the solution to the question."""
        question = self.get_object()
        answer_id = request.data.get("answer")

        try:
            answer = Answer.objects.get(pk=answer_id)
        except Answer.DoesNotExist:
            return Response(
                {"answer": "This field is required."}, status=status.HTTP_400_BAD_REQUEST
            )

        question.set_solution(answer, request.user)
        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=["post"], permission_classes=[permissions.IsAuthenticated])
    def helpful(self, request, pk=None):
        if is_ratelimited(request, "question-vote", "10/d"):
            raise GenericAPIException(
                status.HTTP_429_TOO_MANY_REQUESTS,
                "You've exceeded the number of votes for questions allowed in a day.",
            )

        question = self.get_object()

        if not question.editable:
            raise GenericAPIException(status.HTTP_403_FORBIDDEN, "Question not editable")
        if question.has_voted(request):
            raise GenericAPIException(status.HTTP_409_CONFLICT, "Cannot vote twice")

        QuestionVote(question=question, creator=request.user).save()
        num_votes = QuestionVote.objects.filter(question=question).count()
        return Response({"num_votes": num_votes})

    @action(detail=True, methods=["post"], permission_classes=[permissions.IsAuthenticated])
    def follow(self, request, pk=None):
        question = self.get_object()
        actstream.actions.follow(request.user, question, actor_only=False, send_action=False)
        return Response("", status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=["post"], permission_classes=[permissions.IsAuthenticated])
    def unfollow(self, request, pk=None):
        question = self.get_object()
        actstream.actions.unfollow(request.user, question, send_action=False)
        return Response("", status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=["post"])
    def set_metadata(self, request, pk=None):
        data = {}
        data.update(request.data)
        data["question"] = self.get_object().pk

        serializer = QuestionMetaDataSerializer(data=data)
        if serializer.is_valid():
            serializer.save()
            return Response(serializer.data)
        else:
            return Response(serializer.errors, status=status.HTTP_400_BAD_REQUEST)

    @action(detail=True, methods=["post", "delete"])
    def delete_metadata(self, request, pk=None):
        question = self.get_object()

        if "name" not in request.data:
            return Response(
                {"name": "This field is required."}, status=status.HTTP_400_BAD_REQUEST
            )

        try:
            meta = QuestionMetaData.objects.get(question=question, name=request.data["name"])
            meta.delete()
            return Response(status=status.HTTP_204_NO_CONTENT)
        except QuestionMetaData.DoesNotExist:
            raise GenericAPIException(
                status.HTTP_404_NOT_FOUND, "No matching metadata object found."
            )

    @action(
        detail=True, methods=["post"], permission_classes=[permissions.IsAuthenticatedOrReadOnly]
    )
    def take(self, request, pk=None):
        question = self.get_object()
        field = serializers.BooleanField()
        force = field.to_internal_value(request.data.get("force", False))

        try:
            question.take(request.user, force=force)
        except InvalidUserException:
            raise GenericAPIException(
                status.HTTP_400_BAD_REQUEST, "Question creator cannot take a question."
            )
        except AlreadyTakenException:
            raise GenericAPIException(
                status.HTTP_409_CONFLICT, "Conflict: question is already taken."
            )

        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(
        detail=True,
        methods=["post"],
        permission_classes=[permissions.IsAuthenticated, HasAddTagPermissions],
    )
    def add_tags(self, request, pk=None):
        question = self.get_object()

        if "tags" not in request.data:
            return Response(
                {"tags": "This field is required."}, status=status.HTTP_400_BAD_REQUEST
            )

        tags = request.data["tags"]

        for tag in tags:
            try:
                add_existing_tag(tag, question.tags)
            except SumoTag.DoesNotExist:
                raise GenericAPIException(
                    status.HTTP_403_FORBIDDEN, "You are not authorized to create new tags."
                )

        data = [{"name": tag.name, "slug": tag.slug} for tag in question.tags.all()]
        return Response(data)

    @action(
        detail=True,
        methods=["post", "delete"],
        permission_classes=[permissions.IsAuthenticated, HasRemoveTagPermissions],
    )
    def remove_tags(self, request, pk=None):
        question = self.get_object()

        if "tags" not in request.data:
            return Response(
                {"tags": "This field is required."}, status=status.HTTP_400_BAD_REQUEST
            )

        tags = request.data["tags"]

        for tag in tags:
            question.tags.remove(tag)

        return Response(status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=["post"], permission_classes=[permissions.IsAuthenticated])
    def auto_tag(self, request, pk=None):
        question = self.get_object()
        question.auto_tag()
        return Response(status=status.HTTP_204_NO_CONTENT)


class AnswerSerializer(serializers.ModelSerializer):
    content = SplitSourceField(read_source="content_parsed", write_source="content")
    created = DateTimeUTCField(read_only=True)
    creator = serializers.SerializerMethodField()
    num_helpful_votes = serializers.ReadOnlyField()
    num_unhelpful_votes = serializers.ReadOnlyField()
    updated = DateTimeUTCField(read_only=True)
    updated_by = serializers.SerializerMethodField()

    class Meta:
        model = Answer
        fields = (
            "id",
            "question",
            "content",
            "created",
            "creator",
            "updated",
            "updated_by",
            "is_spam",
            "num_helpful_votes",
            "num_unhelpful_votes",
        )

    def get_creator(self, obj):
        profile = get_or_create_profile(obj.creator)
        return ProfileFKSerializer(profile).data

    def get_updated_by(self, obj):
        profile = get_or_create_profile(obj.updated_by)
        return ProfileFKSerializer(profile).data if profile else None

    def validate(self, data):
        user = self.context.get("request").user
        if user and not user.is_anonymous and data.get("creator") is None:
            data["creator"] = user
        return data


class AnswerFKSerializer(AnswerSerializer):
    class Meta:
        model = Answer
        fields = (
            "id",
            "question",
            "creator",
        )


class AnswerFilter(django_filters.FilterSet):
    creator = django_filters.CharFilter(field_name="creator__username")
    question = django_filters.Filter(field_name="question__id")
    updated = django_filters.IsoDateTimeFilter()
    created = django_filters.IsoDateTimeFilter()

    class Meta:
        model = Answer
        fields = {
            "question": ["exact"],
            "creator": ["exact"],
            "created": ["gt", "lt", "exact"],
            "updated": ["gt", "lt", "exact"],
            "updated_by": ["exact"],
            "is_spam": ["exact"],
        }


class AnswerViewSet(viewsets.ModelViewSet):
    serializer_class = AnswerSerializer
    queryset = Answer.objects.all()
    permission_classes = [
        OnlyCreatorEdits,
        permissions.IsAuthenticatedOrReadOnly,
    ]
    filterset_class = AnswerFilter
    filter_backends = [
        DjangoFilterBackend,
        OrderingFilter,
    ]
    filterset_fields = [
        "question",
        "created",
        "creator",
        "updated",
        "updated_by",
    ]
    ordering_fields = [
        "id",
        "created",
        "updated",
    ]
    # Default, if not overwritten
    ordering = ("-id",)

    def get_pagination_serializer(self, page):
        """
        Return a serializer instance to use with paginated data.
        """

        class SerializerClass(self.pagination_serializer_class):
            class Meta:
                object_serializer_class = AnswerSerializer

        context = self.get_serializer_context()
        return SerializerClass(instance=page, context=context)

    @action(detail=True, methods=["post"], permission_classes=[permissions.IsAuthenticated])
    def helpful(self, request, pk=None):
        if is_ratelimited(request, "answer-vote", "10/d"):
            raise GenericAPIException(
                status.HTTP_429_TOO_MANY_REQUESTS,
                "You've exceeded the number of votes for answers allowed in a day.",
            )

        answer = self.get_object()

        if not answer.question.editable:
            raise GenericAPIException(status.HTTP_403_FORBIDDEN, "Answer not editable")
        if answer.has_voted(request):
            raise GenericAPIException(status.HTTP_409_CONFLICT, "Cannot vote twice")

        AnswerVote(answer=answer, creator=request.user, helpful=True).save()
        num_helpful_votes = AnswerVote.objects.filter(answer=answer, helpful=True).count()
        num_unhelpful_votes = AnswerVote.objects.filter(answer=answer, helpful=False).count()
        return Response(
            {
                "num_helpful_votes": num_helpful_votes,
                "num_unhelpful_votes": num_unhelpful_votes,
            }
        )

    @action(detail=True, methods=["post"], permission_classes=[permissions.IsAuthenticated])
    def follow(self, request, pk=None):
        answer = self.get_object()
        actstream.actions.follow(request.user, answer, actor_only=False)
        return Response("", status=status.HTTP_204_NO_CONTENT)

    @action(detail=True, methods=["post"], permission_classes=[permissions.IsAuthenticated])
    def unfollow(self, request, pk=None):
        answer = self.get_object()
        actstream.actions.unfollow(request.user, answer)
        return Response("", status=status.HTTP_204_NO_CONTENT)
