# Generated by Django 4.2.16 on 2024-10-15 04:43

from re import A

import django.db.models.deletion
from django.db import migrations, models

import kitsune.sumo.fields

# The key is the topic slug to migrate to and the value is the list
# of topic slugs to migrate from,
# If the topic for each product does not exist, it will be created
# If it exists, the articles will be moved to the existing topic for each product.
# If the topics is created, a redirect will be added to the old topic.
TOPICS_TO_MIGRATE = {
    "install-and-update": [
        "install-and-update-firefox",
        "install-and-update-firefox-ios",
        "download-and-install",
        "download-setup",
        "deploy-firefox-for-enterprise",
        "installation-enterprise",
        "deploy-firefox-enterprise-mac",
        "deploy-firefox-enterprise-windows",
        "deploy-firefox-enterprise-linux",
        "install-migrate-and-update",
        "download-install-and-migration",
        "install-lockwise",
    ],
    "troubleshooting": [
        "fix-problems",
        "fix-problems-firefox",
        "crashes-errors-and-other-issues",
        "firefox-ios-not-working-expected",
        "fix-slowness-crashing-error-messages-and-other-problems",
        "troubleshoot",
        "troubleshoot-issues",
        "fix-problems-lockwise",
        "troubleshooting-reality",
    ],
    "protect-your-privacy": [
        "stay-private-and-secure",
        "privacy-and-security",
        "privacy-and-security-settings",
        "passwords-forms-and-search",
        "security-privacy",
        "protect-your-privacy-firefox-android",
        "privacy-settings-firefox-ios",
    ],
    "emails": [
        "emails-thunderbird",
        "set-up-email-thunderbird",
    ],
    "customize-settings-and-preferences": [
        "manage-preferences-and-add-ons-firefox",
        "firefox-settings",
        "personalize-firefox",
        "personalize-firefox-settings",
        "customize",
        "controls",
        "tab-settings",
        "firefox-options-preferences-and-settings",
        "bookmark-options",
        "personalize-your-experience",
        "firefox-android-settings",
        "customize-settings-firefox-android",
        "customize-preferences-ios",
        "customization-firefox-enterprise-environment",
        "policies-customization-enterprise",
        "manage-certificates-firefox-enterprise",
        "policies-overview-enterprise",
        "manage-settings-policy",
        "autoconfiguration-enterprise",
        "hubs-permissions-settings",
        "thunderbird-controls-and-buttons",
        "tab-settings",
        "thunderbird-options-preferences-and-settings",
        "update-account-settings",
        "customize-controls-options-and-add-ons",
    ],
    "install-and-manage-add-ons": [
        "install-and-manage-add-ons-firefox",
        "use-extensions-and-add-ons",
        "add-ons",
        "manage-add-ons-enterprise",
        "customize-thunderbird-add-ons-plugins-and-extensions",
    ],
}

NEW_TOPICS_DATA = [
    {
        "title": "Backup, recovery, and sync",
        "slug": "backup-recovery-and-sync",
        "description": "Sync your data across different platforms and devices, backup what’s important, and recover it if it’s lost.",
        "products": ["firefox", "mobile", "ios", "mozilla-account"],
        "subtopics": [
            {
                "title": "Backup data",
                "slug": "backup-data",
                "description": "From passwords to bookmarks, discover ways to back up your important data.",
                "products": ["firefox", "mobile", "ios", "mozilla-account"],
            },
            {
                "title": "Recover data",
                "slug": "recover-data",
                "description": "Learn how to recover your lost data.",
                "products": ["firefox", "mobile", "ios", "mozilla-account"],
            },
            {
                "title": "Sync data",
                "slug": "sync-data",
                "description": "Sync your data across different platforms and devices.",
                "products": ["firefox", "mobile", "ios", "mozilla-account"],
            },
        ],
    },
    {
        "title": "Browse",
        "slug": "browse",
        "description": "Explore how to navigate the web efficiently and effectively with Mozilla’s products.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "pocket",
            "hubs",
            "focus-firefox",
            "firefox-enterprise",
        ],
        "subtopics": [
            {
                "title": "Article View",
                "slug": "article-view",
                "description": "Comfortably read web pages without distractions.",
                "products": [],
            },
            {
                "title": "Tabs",
                "slug": "tabs",
                "description": "Organize and manage your tabs for better browsing.",
                "products": ["firefox", "mobile", "ios"],
            },
            {
                "title": "History",
                "slug": "history",
                "description": "View and manage your browsing history.",
                "products": ["firefox", "mobile", "ios"],
            },
            {
                "title": "Bookmarks",
                "slug": "bookmarks",
                "description": "Save and organize your favorite web content with bookmarks.",
                "products": ["firefox", "mobile", "ios"],
            },
            {
                "title": "Audio and Video",
                "slug": "audio-and-video",
                "description": "Everything you need to know about audio and video.",
                "products": ["firefox", "mobile", "ios"],
            },
            {
                "title": "Images and documents",
                "slug": "images-and-documents",
                "description": "Everything you need to know about images and documents.",
                "products": ["firefox", "mobile", "ios"],
            },
            {
                "title": "Links",
                "slug": "links",
                "description": "Use web links in Mozilla’s products.",
                "products": ["firefox", "mobile", "ios", "focus-firefox"],
            },
            {
                "title": "Home screen",
                "slug": "home-screen",
                "description": "Add Mozilla’s products to your home screen for easy access.",
                "products": ["firefox", "mobile", "ios"],
            },
        ],
    },
    {
        "title": "Performance and connectivity",
        "slug": "performance-and-connectivity",
        "description": "Deal with error messages, crashing applications, connectivity issues, and slow performance.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "firefox-private-network-vpn",
            "pocket",
            "mdn-plus",
            "focus-firefox",
            "thunderbird",
            "thunderbird-android",
            "firefox-enterprise",
            "relay",
        ],
        "subtopics": [
            {
                "title": "Crashing and slow performance",
                "slug": "crashing-and-slow-performance",
                "description": "Troubleshoot issues with app crashes or slow performance.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "pocket",
                    "focus-firefox",
                    "thunderbird",
                    "thunderbird-android",
                    "firefox-enterprise",
                    "relay",
                ],
            },
            {
                "title": "Connectivity",
                "slug": "connectivity",
                "description": "Troubleshoot issues with connectivity.",
                "products": [
                    "firefox-private-network-vpn",
                    "thunderbird",
                    "thunderbird-android",
                ],
            },
            {
                "title": "Error codes",
                "slug": "error-codes",
                "description": "Troubleshoot issues with error codes.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                ],
            },
            {
                "title": "Site breakages",
                "slug": "site-breakages",
                "description": "Resolve problems when websites don’t load or function correctly.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "focus-firefox",
                    "firefox-private-network-vpn",
                ],
            },
        ],
    },
    {
        "title": "Accessibility",
        "slug": "accessibility",
        "description": "Learn about Mozilla products’ accessibility features and settings for better usability.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "pocket",
            "thunderbird",
            "thunderbird-android",
            "mozilla-account",
        ],
        "subtopics": [
            {
                "title": "Reader View",
                "slug": "reader-view",
                "description": "Comfortably read web pages without distractions.",
                "products": ["firefox", "mobile", "ios"],
            },
            {
                "title": "Text-to-speech",
                "slug": "text-to-speech",
                "description": "Read web pages aloud.",
                "products": ["firefox", "mobile", "ios", "pocket"],
            },
        ],
    },
    {
        "title": "Download and save",
        "slug": "download-and-save",
        "description": "Manage your downloaded files and saved content",
        "products": ["firefox", "mobile", "ios", "pocket", "mdn-plus"],
        "subtopics": [
            {
                "title": "Downloads",
                "slug": "downloads",
                "description": "Manage and troubleshoot your downloads.",
                "products": ["firefox", "mobile", "ios", "mdn-plus"],
            },
            {
                "title": "Save content",
                "slug": "save-content",
                "description": "Save your favorite content for later.",
                "products": ["firefox", "mobile", "ios", "pocket", "mdn-plus"],
            },
        ],
    },
    {
        "title": "Search, tag, and share",
        "slug": "search-tag-and-share",
        "description": "Learn about search functionality and how to organize or share content in Mozilla products.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "pocket",
            "focus-firefox",
            "thunderbird",
            "thunderbird-android",
        ],
        "subtopics": [
            {
                "title": "Search",
                "slug": "search",
                "description": "Find information quickly and easily.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "pocket",
                    "focus-firefox",
                    "thunderbird",
                    "thunderbird-android",
                ],
            },
            {
                "title": "Tags",
                "slug": "tags",
                "description": "Categorize and organize efficiently with tags.",
                "products": [
                    "pocket",
                    "thunderbird",
                    "thunderbird-android",
                ],
            },
            {
                "title": "Share content",
                "slug": "share-content",
                "description": "Learn how to share links, content, messages, and more.",
                "products": ["firefox", "mobile", "ios", "pocket"],
            },
        ],
    },
    {
        "title": "Installation and updates",
        "slug": "installation-and-updates",
        "description": "Learn how to install your favorite Mozilla products and keep them updated.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "firefox-private-network-vpn",
            "pocket",
            "focus-firefox",
            "firefox-enterprise",
            "thunderbird",
            "thunderbird-android",
        ],
        "subtopics": [
            {
                "title": "Install",
                "slug": "install",
                "description": "Learn about how to install Mozilla’s products on your devices.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "pocket",
                    "focus-firefox",
                    "firefox-enterprise",
                    "thunderbird",
                    "thunderbird-android",
                ],
            },
            {
                "title": "Update",
                "slug": "update",
                "description": "Keep the Mozilla products you use up to date with the latest features and improvements.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "pocket",
                    "focus-firefox",
                    "firefox-enterprise",
                    "thunderbird",
                    "thunderbird-android",
                ],
            },
        ],
    },
    {
        "title": "Settings",
        "slug": "settings",
        "description": "Manage and customize your product experience with settings, add-ons, and more.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "firefox-private-network-vpn",
            "relay",
            "monitor",
            "pocket",
            "mdn-plus",
            "focus-firefox",
            "firefox-enterprise",
            "thunderbird",
            "thunderbird-android",
            "mozilla-account",
        ],
        "subtopics": [
            {
                "title": "Import and export settings",
                "slug": "import-and-export-settings",
                "description": "Learn how to import and export product settings for easy setup and migration.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "focus-firefox",
                    "firefox-enterprise",
                    "thunderbird",
                    "thunderbird-android",
                ],
            },
            {
                "title": "Customization",
                "slug": "customization",
                "description": "Customize product appearance and functionality to suit your preferences.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "pocket",
                    "focus-firefox",
                    "firefox-enterprise",
                    "thunderbird",
                    "thunderbird-android",
                    "mdn-plus",
                ],
            },
            {
                "title": "Autofill",
                "slug": "autofill",
                "description": "Automatically fill your personal information into form fields and more.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "focus-firefox",
                    "firefox-enterprise",
                    "thunderbird",
                    "thunderbird-android",
                ],
            },
            {
                "title": "Languages",
                "slug": "languages",
                "description": "Change and manage language settings in Mozilla’s products.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "relay",
                    "focus-firefox",
                    "firefox-enterprise",
                    "thunderbird",
                    "thunderbird-android",
                ],
            },
            {
                "title": "Add-ons, extensions, and themes",
                "slug": "add-ons-extensions-and-themes",
                "description": "Enhance product functionality with add-ons, extensions, and themes.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "relay",
                    "pocket",
                    "focus-firefox",
                    "firefox-enterprise",
                    "thunderbird",
                    "mozilla-account",
                ],
            },
            {
                "title": "Notifications",
                "slug": "notifications",
                "description": "Manage notifications in Mozilla’s products.",
                "products": ["firefox", "mobile", "ios"],
            },
        ],
    },
    {
        "title": "Privacy and security",
        "slug": "privacy-and-security",
        "description": "Learn how to protect your privacy and secure your data.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "firefox-private-network-vpn",
            "relay",
            "focus-firefox",
            "firefox-enterprise",
            "monitor",
            "pocket",
            "thunderbird",
            "thunderbird-android",
            "mozilla-account",
        ],
        "subtopics": [
            {
                "title": "Data removal",
                "slug": "data-removal",
                "description": "Remove your data from Mozilla’s products.",
                "products": ["monitor"],
            },
            {
                "title": "Tracking protection",
                "slug": "tracking-protection",
                "description": "Enable tracking protection to enhance your privacy.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "focus-firefox",
                    "firefox-enterprise",
                    "mozilla-account",
                    "relay",
                    "pocket",
                ],
            },
            {
                "title": "Encryption",
                "slug": "encryption",
                "description": "Use encryption features to keep yourself and your data secure.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "focus-firefox",
                    "firefox-enterprise",
                    "thunderbird",
                    "thunderbird-android",
                    "mozilla-account",
                ],
            },
            {
                "title": "Masking",
                "slug": "masking",
                "description": "Mask your email and phone number to better protect your privacy.",
                "products": ["relay", "mozilla-account"],
            },
            {
                "title": "Security",
                "slug": "security",
                "description": "Keep yourself secure online.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "relay",
                    "focus-firefox",
                    "firefox-enterprise",
                    "monitor",
                    "thunderbird",
                    "thunderbird-android",
                    "mozilla-account",
                ],
            },
        ],
    },
    {
        "title": "Passwords and sign in",
        "slug": "passwords-and-sign-in",
        "description": "Manage your passwords and securely access your accounts.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "firefox-private-network-vpn",
            "relay",
            "monitor",
            "mdn-plus",
            "hubs",
            "thunderbird",
            "thunderbird-android",
            "mozilla-account",
            "pocket",
        ],
        "subtopics": [
            {
                "title": "Save passwords",
                "slug": "save-passwords",
                "description": "Save your passwords for easy sign-in.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "thunderbird",
                    "thunderbird-android",
                    "mozilla-account",
                    "monitor",
                ],
            },
            {
                "title": "Sign in",
                "slug": "sign-in",
                "description": "Sign in to your accounts securely.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                ],
            },
            {
                "title": "Reset passwords",
                "slug": "reset-passwords",
                "description": "Reset your passwords when you forget them.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "thunderbird",
                    "thunderbird-android",
                    "mozilla-account",
                    "pocket",
                ],
            },
            {
                "title": "Two-factor authentication",
                "slug": "two-factor-authentication",
                "description": "Use two-factor authentication to add an extra layer of security when signing in.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "mozilla-account",
                ],
            },
            {
                "title": "Primary password",
                "slug": "primary-password",
                "description": "Set up a primary password to secure your saved passwords.",
                "products": ["firefox", "mobile", "ios", "firefox-private-network-vpn"],
            },
        ],
    },
    {
        "title": "Accounts",
        "slug": "accounts",
        "description": "Manage your accounts and profiles for Mozilla’s products and services.",
        "products": [
            "firefox",
            "mobile",
            "ios",
            "firefox-private-network-vpn",
            "pocket",
            "hubs",
            "firefox-enterprise",
            "thunderbird",
            "thunderbird-android",
            "mozilla-account",
            "mdn-plus",
        ],
        "subtopics": [
            {
                "title": "Profiles",
                "slug": "profiles",
                "description": "Create and manage user profiles.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "pocket",
                    "firefox-enterprise",
                    "thunderbird",
                    "thunderbird-android",
                    "mozilla-account",
                ],
            },
            {
                "title": "Account management",
                "slug": "account-management",
                "description": "Manage your account settings and preferences.",
                "products": [
                    "firefox",
                    "mobile",
                    "ios",
                    "firefox-private-network-vpn",
                    "pocket",
                    "thunderbird",
                    "thunderbird-android",
                    "mozilla-account",
                ],
            },
            {
                "title": "Account recovery",
                "slug": "account-recovery",
                "description": "Recover your account if you lose access to it or encounter issues.",
                "products": [
                    "firefox-private-network-vpn",
                    "pocket",
                    "mozilla-account",
                    "mdn-plus",
                ],
            },
        ],
    },
    {
        "title": "Billing and subscriptions",
        "slug": "billing-and-subscriptions",
        "description": "Manage your subscriptions and payment details for Mozilla’s premium products.",
        "products": [
            "firefox-private-network-vpn",
            "relay",
            "monitor",
            "pocket",
            "mdn-plus",
            "hubs",
            "mozilla-account",
        ],
        "subtopics": [
            {
                "title": "Manage billing",
                "slug": "manage-billing",
                "description": "Manage your payment details and billing information.",
                "products": [
                    "firefox-private-network-vpn",
                    "relay",
                    "pocket",
                    "mdn-plus",
                    "hubs",
                    "mozilla-account",
                    "monitor",
                ],
            },
            {
                "title": "Manage subscriptions",
                "slug": "manage-subscriptions",
                "description": "Manage your subscriptions to Mozilla’s premium products and services.",
                "products": [
                    "firefox-private-network-vpn",
                    "relay",
                    "pocket",
                    "mdn-plus",
                    "hubs",
                    "mozilla-account",
                    "monitor",
                ],
            },
        ],
    },
    {
        "title": "Email and messaging",
        "slug": "email-and-messaging",
        "description": "Read, send, and manage your emails and messages.",
        "products": ["thunderbird", "thunderbird-android", "relay"],
        "subtopics": [
            {
                "title": "Contacts",
                "slug": "contacts",
                "description": "Manage your email contacts.",
                "products": ["thunderbird", "thunderbird-android"],
            },
            {
                "title": "Junk email and spam",
                "slug": "junk-mail-and-spam",
                "description": "Identify and manage junk email and spam.",
                "products": ["thunderbird", "thunderbird-android"],
            },
            {
                "title": "Instant messaging",
                "slug": "instant-messaging",
                "description": "Features and functionality to use with instant messaging.",
                "products": ["thunderbird", "thunderbird-android", "relay"],
            },
            {
                "title": "Attachments",
                "slug": "attachments",
                "description": "Learn how to use and manage email attachments.",
                "products": ["thunderbird", "thunderbird-android"],
            },
            {
                "title": "Send and receive email",
                "slug": "send-and-receive-email",
                "description": "Everything you need to know about sending and receiving emails.",
                "products": ["thunderbird", "thunderbird-android", "relay"],
            },
            {
                "title": "Import and export email",
                "slug": "import-and-export-email",
                "description": "Import and export your emails for backup or transfer.",
                "products": ["thunderbird", "thunderbird-android"],
            },
            {
                "title": "Calendar",
                "slug": "calendar",
                "description": "Manage your calendar events and schedules.",
                "products": ["thunderbird", "thunderbird-android"],
            },
        ],
    },
]

TAXONOMY_MAPPING = [
    {
        "old_title": "Fix problems with websites",
        "old_slug": "websites",
        "title": "Site breakages",
        "slug": "site-breakages",
        "parent_topic_slug": "performance-and-connectivity",
    },
    {
        "old_title": "Firefox crashes",
        "old_slug": "crashing",
        "title": "Crashing and slow performance",
        "slug": "crashing-and-slow-performance",
        "parent_topic_slug": "performance-and-connectivity",
    },
    {
        "old_title": "Unblock Firefox from connecting to the internet",
        "old_slug": "unblock-firefox-connecting-internet",
        "title": "Performance and connectivity",
        "slug": "performance-and-connectivity",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Firefox won't save settings or remember information",
        "old_slug": "firefox-wont-save-settings-or-remember-information",
        "title": "Settings",
        "slug": "settings",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Problems with add-ons, plugins or unwanted software",
        "old_slug": "problems-add-ons-plugins-or-unwanted-software",
        "title": "Add-ons, extensions, and themes",
        "slug": "add-ons-extensions-and-themes",
        "parent_topic_slug": "settings",
    },
    {
        "old_title": "Videos, sound, pictures and animations don't work",
        "old_slug": "videos-sound-pictures-and-animations-dont-work",
        "title": "Audio and video",
        "slug": "audio-and-video",
        "parent_topic_slug": "browse",
    },
    {
        "old_title": "Basic browsing",
        "old_slug": "basic-browsing-firefox",
        "title": "Browse",
        "slug": "browse",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Basic browsing",
        "old_slug": "basic-browsing-firefox-android",
        "title": "Browse",
        "slug": "browse",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Basic browsing",
        "old_slug": "basic-browsing-firefox-ios",
        "title": "Browse",
        "slug": "browse",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Bookmarks and Tab",
        "old_slug": "bookmarks",
        "title": "Bookmarks",
        "slug": "bookmarks",
        "parent_topic_slug": "browse",
    },
    {
        "old_title": "Bookmarks",
        "old_slug": "back-up-bookmarks",
        "title": "Bookmarks",
        "slug": "bookmarks",
        "parent_topic_slug": "browse",
    },
    {
        "old_title": "Bookmarks and Tabs",
        "old_slug": "bookmarks-and-tabs-firefox-ios",
        "title": "Bookmarks",
        "slug": "bookmarks",
        "parent_topic_slug": "browse",
    },
    {
        "old_title": "Reader View and List",
        "old_slug": "reader-view-and-list",
        "title": "Browse",
        "slug": "browse",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Install and update",
        "old_slug": "install-and-update",
        "title": "Installation and updates",
        "slug": "installation-and-updates",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Sync and save",
        "old_slug": "sync-and-save",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Firefox Sync settings",
        "old_slug": "sync",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Customize settings and preferences",
        "old_slug": "customize-settings-and-preferences",
        "title": "Settings",
        "slug": "settings",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Import settings from other browsers",
        "old_slug": "import-settings-other-browsers-firefox",
        "title": "Import and export settings",
        "slug": "import-and-export-settings",
        "parent_topic_slug": "settings",
    },
    {
        "old_title": "Video, audio and interactive settings",
        "old_slug": "video-audio-and-file-settings-Firefox",
        "title": "Multimedia",
        "slug": "multimedia",
        "parent_topic_slug": "browse",
    },
    {
        "old_title": "Display and appearance",
        "old_slug": "customize-firefox-appearance",
        "title": "Customization",
        "slug": "customization",
        "parent_topic_slug": "settings",
    },
    {
        "old_title": "Install and manage add-ons",
        "old_slug": "install-and-manage-add-ons",
        "title": "Add-ons, extensions, and themes",
        "slug": "add-ons-extensions-and-themes",
        "parent_topic_slug": "settings",
    },
    {
        "old_title": "Search",
        "old_slug": "ios-search",
        "title": "Search",
        "slug": "search",
        "parent_topic_slug": "search-tag-and-share",
    },
    {
        "old_title": "Protect your privacy",
        "old_slug": "protect-your-privacy",
        "title": "Privacy and security",
        "slug": "privacy-and-security",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Cookies and cache",
        "old_slug": "cookies",
        "title": "Tracking protection",
        "slug": "tracking-protection",
        "parent_topic_slug": "privacy-and-security",
    },
    {
        "old_title": "Privacy and security",
        "old_slug": "privacy-and-security",
        "title": "Privacy and security",
        "slug": "privacy-and-security",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Tracking protection",
        "old_slug": "tracking-protection",
        "title": "Tracking protection",
        "slug": "tracking-protection",
        "parent_topic_slug": "privacy-and-security",
    },
    {
        "old_title": "Firefox for families",
        "old_slug": "firefox-for-families",
        "title": "Privacy and security",
        "slug": "privacy-and-security",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Use extension and add-ons",
        "old_slug": "use-extensions-and-add-ons",
        "title": "Add-ons, extensions, and themes",
        "slug": "add-ons-extensions-and-themes",
        "parent_topic_slug": "settings",
    },
    {
        "old_title": "Stay connected",
        "old_slug": "stay-connected",
        "title": "Performance and connectivity",
        "slug": "performance-and-connectivity",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Personalize your experience",
        "old_slug": "personalize-your-experience",
        "title": "Customization",
        "slug": "customization",
        "parent_topic_slug": "settings",
    },
    {
        "old_title": "Back up your data",
        "old_slug": "back-up-your-data",
        "title": "Backup data",
        "slug": "backup-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Profile",
        "old_slug": "back-up-profile",
        "title": "Account management",
        "slug": "account-management",
        "parent_topic_slug": "accounts",
    },
    {
        "old_title": "Sync",
        "old_slug": "setup-sync",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Sync and save",
        "old_slug": "save-share-and-sync-firefox-android",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Sync, share and save",
        "old_slug": "sync-share-save",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Sync and save",
        "old_slug": "sync-and-save",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Firefox Sync",
        "old_slug": "sync",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Sync and save",
        "old_slug": "save-and-share-firefox-ios",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Sync devices & services",
        "old_slug": "sync-devices-services",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Sync",
        "old_slug": "sync-preview",
        "title": "Sync data",
        "slug": "sync-data",
        "parent_topic_slug": "backup-recovery-and-sync",
    },
    {
        "old_title": "Privacy and security settings",
        "old_slug": "privacy-and-security-settings",
        "title": "Privacy and security",
        "slug": "privacy-and-security",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Passwords, forms and search",
        "old_slug": "passwords-forms-and-search",
        "title": "Passwords and sign in",
        "slug": "passwords-and-sign-in",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Thunderbird controls and buttons",
        "old_slug": "thunderbird-controls-and-buttons",
        "title": "Settings",
        "slug": "settings",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Calendar",
        "old_slug": "calendar",
        "title": "Calendar",
        "slug": "calendar",
        "parent_topic_slug": "email-and-messaging",
    },
    {
        "old_title": "Thunderbird versions and languages",
        "old_slug": "thunderbird-versions-and-languages",
        "title": "Settings",
        "slug": "settings",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Copy your personal information from one Thunderbird to another",
        "old_slug": "copy-your-personal-information-one-thunderbird-another",
        "title": "Import and export email",
        "slug": "import-and-export-email",
        "parent_topic_slug": "email-and-messaging",
    },
    {
        "old_title": "Emails",
        "old_slug": "emails",
        "title": "Email and messaging",
        "slug": "email-and-messaging",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Set up email",
        "old_slug": "set-up-email-thunderbird",
        "title": "Account management",
        "slug": "account-management",
        "parent_topic_slug": "accounts",
    },
    {
        "old_title": "Read, send and organize emails",
        "old_slug": "read-send-and-organize-emails",
        "title": "Send and receive email",
        "slug": "send-and-receive-email",
        "parent_topic_slug": "email-and-messaging",
    },
    {
        "old_title": "News Feeds (RSS), Blogs and Social",
        "old_slug": "news-feeds-rss-blogs-and-social-thunderbird",
        "title": "Settings",
        "slug": "settings",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Contacts",
        "old_slug": "contacts-thunderbird",
        "title": "Contacts",
        "slug": "contacts",
        "parent_topic_slug": "email-and-messaging",
    },
    {
        "old_title": "Accounts & Sign-in",
        "old_slug": "accounts",
        "title": "Accounts",
        "slug": "accounts",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Billing & Payments",
        "old_slug": "payments",
        "title": "Manage billing",
        "slug": "manage-billing",
        "parent_topic_slug": "billing-and-subscriptions",
    },
    {
        "old_title": "Subscriptions & billing",
        "old_slug": "subscriptions-billing",
        "title": "Manage subscriptions",
        "slug": "manage-subscriptions",
        "parent_topic_slug": "billing-and-subscriptions",
    },
    {
        "old_title": "Manage subscriptions",
        "old_slug": "manage-subscriptions",
        "title": "Manage subscriptions",
        "slug": "manage-subscriptions",
        "parent_topic_slug": "billing-and-subscriptions",
    },
    {
        "old_title": "Billing & Payments",
        "old_slug": "subscriptions-and-payments",
        "title": "Manage billing",
        "slug": "manage-billing",
        "parent_topic_slug": "billing-and-subscriptions",
    },
    {
        "old_title": "Services and subscriptions",
        "old_slug": "services-and-subscriptions",
        "title": "Manage subscriptions",
        "slug": "manage-subscriptions",
        "parent_topic_slug": "billing-and-subscriptions",
    },
    {
        "old_title": "Connectivity",
        "old_slug": "connectivity",
        "title": "Performance and connectivity",
        "slug": "performance-and-connectivity",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Your Pocket Account",
        "old_slug": "your-pocket-account",
        "title": "Accounts",
        "slug": "accounts",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Pocket Premium",
        "old_slug": "pocket-premium",
        "title": "Manage subscriptions",
        "slug": "manage-subscriptions",
        "parent_topic_slug": "billing-and-subscriptions",
    },
    {
        "old_title": "Pocket Browser Extensions",
        "old_slug": "pocket-browser-extensions",
        "title": "Add-ons, extensions, and themes",
        "slug": "add-ons-extensions-and-themes",
        "parent_topic_slug": "settings",
    },
    {
        "old_title": "Free email masking",
        "old_slug": "free-email-masking",
        "title": "Masking",
        "slug": "masking",
        "parent_topic_slug": "privacy-and-security",
    },
    {
        "old_title": "Premium email masking",
        "old_slug": "premium-email-masking",
        "title": "Masking",
        "slug": "masking",
        "parent_topic_slug": "privacy-and-security",
    },
    {
        "old_title": "Premium phone masking",
        "old_slug": "premium-phone-masking",
        "title": "Masking",
        "slug": "masking",
        "parent_topic_slug": "privacy-and-security",
    },
    {
        "old_title": "Accounts & Sign-in",
        "old_slug": "accounts-and-login",
        "title": "Accounts",
        "slug": "accounts",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Accounts",
        "old_slug": "accounts",
        "title": "Accounts",
        "slug": "accounts",
        "parent_topic_slug": "",
    },
    {
        "old_title": "Billing & Payments",
        "old_slug": "billing-payments",
        "title": "Manage billing",
        "slug": "manage-billing",
        "parent_topic_slug": "billing-and-subscriptions",
    },
    {
        "old_title": "Monitor",
        "old_slug": "monitoring",
        "title": "Privacy and security",
        "slug": "privacy-and-security",
        "parent_topic_slug": "",
    },
]


def migrate_topics(apps, schema_editor):
    """Migrate topics to the new taxonomy."""

    Topic = apps.get_model("products", "Topic")
    Product = apps.get_model("products", "Product")
    TopicSlugHistory = apps.get_model("products", "TopicSlugHistory")
    ProductTopic = apps.get_model("products", "ProductTopic")

    # Promote topics for the community product to the new taxonomy
    try:
        product = Product.objects.get(slug="contributor")
    except Product.DoesNotExist:
        ...
    else:
        topics = Topic.objects.filter(product=product)

        for topic in topics:
            topic.products.remove(product)
            topic.products.add(product)

    for entry in TAXONOMY_MAPPING:
        old_topics = Topic.objects.filter(slug=entry["old_slug"], product__isnull=False)
        try:
            new_topic = Topic.objects.get(slug=entry["slug"], product__isnull=True)
        except Topic.DoesNotExist:
            continue

        # we may have a list of topics with the same slug
        for old_topic in old_topics:
            product = old_topic.product
            if not ProductTopic.objects.filter(product=product, topic=new_topic).exists():
                ProductTopic.objects.create(product=product, topic=new_topic)

            # Migrate KB
            documents = old_topic.document_set.all()
            for document in documents:
                document.topics.add(new_topic)
                document.topics.remove(old_topic)
            # Migrate AAQ
            if old_topic.questions.exists():
                for question in old_topic.questions.all():
                    question.topic = new_topic
                    question.save()

            if product.visible:
                topic_slug, created = TopicSlugHistory.objects.get_or_create(
                    slug=entry["old_slug"], defaults={"topic": new_topic}
                )
                if not created:
                    if topic_slug.slug == new_topic.slug:
                        topic_slug.delete()
                    else:
                        topic_slug.topic = new_topic
                        topic_slug.save()

            # Archive the old topic
            old_topic.is_archived = True
            old_topic.save()


def migrate_topics_backwards(apps, schema_editor): ...


def create_new_taxonomy_topics(apps, schema_editor):
    Topic = apps.get_model("products", "Topic")
    Product = apps.get_model("products", "Product")
    ProductTopic = apps.get_model("products", "ProductTopic")

    for topic_data in NEW_TOPICS_DATA:
        new_topic = Topic.objects.create(
            slug=topic_data["slug"],
            title=topic_data["title"],
            description=topic_data["description"],
            display_order=50,
            visible=True,
        )

        for product_slug in topic_data["products"]:
            try:
                product = Product.objects.get(slug=product_slug)
            except Product.DoesNotExist:
                continue
            else:
                ProductTopic.objects.get_or_create(product=product, topic=new_topic)

        for subtopic_data in topic_data["subtopics"]:
            new_subtopic = Topic.objects.create(
                slug=subtopic_data["slug"],
                title=subtopic_data["title"],
                description=subtopic_data["description"],
                visible=True,
                parent=new_topic,
                display_order=60,
            )

            for product_slug in subtopic_data["products"]:
                try:
                    product = Product.objects.get(slug=product_slug)
                except Product.DoesNotExist:
                    continue
                else:
                    ProductTopic.objects.get_or_create(product=product, topic=new_subtopic)


def create_new_taxonomy_topics_backwards(apps, schema_editor):
    pass


def archive_old_topics(apps, schema_editor):
    Topic = apps.get_model("products", "Topic")
    TopicSlugHistory = apps.get_model("products", "TopicSlugHistory")

    for topic_slug in TopicSlugHistory.objects.all():
        try:
            old_topic = Topic.objects.get(slug=topic_slug.slug, product=topic_slug.topic.product)
            old_topic.is_archived = True
            old_topic.save()
        except Topic.DoesNotExist:
            pass


def archive_old_topics_backwards(apps, schema_editor):
    pass


def move_kb_articles(apps, schema_editor):
    Topic = apps.get_model("products", "Topic")
    TopicSlugHistory = apps.get_model("products", "TopicSlugHistory")
    title = ""
    for target_slug in TOPICS_TO_MIGRATE.keys():

        match target_slug:
            case "install-and-update":
                title = "Install and update"
            case "troubleshooting":
                title = "Troubleshooting"
            case "protect-your-privacy":
                title = "Protect your privacy"
            case "emails":
                title = "Emails"
            case "customize-settings-and-preferences":
                title = "Customize settings and preferences"
            case "install-and-manage-add-ons":
                title = "Install and manage add-ons"
            case _:
                pass

        source_topics = Topic.objects.filter(slug__in=TOPICS_TO_MIGRATE.get(target_slug, []))
        for source_topic in source_topics:
            product = source_topic.product
            created = False

            try:
                target_topic = Topic.objects.get(slug=target_slug, product=product)
            except Topic.DoesNotExist:
                created = True
                target_topic = Topic.objects.create(
                    title=title,
                    slug=target_slug,
                    product=product,
                    display_order=source_topic.display_order,
                    visible=True,
                    description=title,
                )

            # Make sure that target topics are not sub-topics
            if target_topic.parent:
                target_topic.parent = None

            # make sure the target topic is visible and it has the correct title
            if not created:
                target_topic.title = title
                target_topic.description = title
                target_topic.visible = True
            target_topic.save()

            # Migrate KB articles to new topics
            documents = source_topic.document_set.all()
            for document in documents:
                document.topics.add(target_topic)
                document.topics.remove(source_topic)

            # Migrate questions to the new topic
            if source_topic.questions.exists():
                for question in source_topic.questions.all():
                    question.topic = target_topic
                    question.save()

            # handle redirects for the old topics
            if (
                not TopicSlugHistory.objects.filter(slug=source_topic.slug).exists()
                and product.visible
            ):
                TopicSlugHistory.objects.create(slug=source_topic.slug, topic=target_topic)

            # Finally let's make the source topic invisible
            source_topic.visible = False
            source_topic.save()


def move_kb_articles_backwards(apps, schema_editor):
    pass


def copy_product_to_products(apps, schema_editor):
    Topic = apps.get_model("products", "Topic")
    for topic in Topic.objects.all():
        if topic.product:
            topic.products.add(topic.product)


def copy_product_to_products_backwards(apps, schema_editor): ...


# Functions from the following migrations need manual copying.
# Move them and any dependencies into this file, then update the
# RunPython operations to refer to the local versions:
# kitsune.products.migrations.0008_auto_20240606_2111
# kitsune.products.migrations.0010_auto_20240624_0616
# kitsune.products.migrations.0012_auto_20240701_0307
# kitsune.products.migrations.0014_auto_20240718_0634
# kitsune.products.migrations.0015_auto_20240722_0449


class Migration(migrations.Migration):

    replaces = [
        ("products", "0001_squashed_0005_auto_20200629_0826"),
        ("products", "0006_alter_product_and_topic_images"),
        ("products", "0007_producttopic_topic_products_and_more"),
        ("products", "0008_auto_20240606_2111"),
        ("products", "0009_topicslughistory"),
        ("products", "0010_auto_20240624_0616"),
        ("products", "0011_product_is_archived_topic_is_archived"),
        ("products", "0012_auto_20240701_0307"),
        ("products", "0013_alter_topic_product"),
        ("products", "0014_auto_20240718_0634"),
        ("products", "0015_auto_20240722_0449"),
        ("products", "0016_topic_in_nav_alter_topic_product"),
        ("products", "0017_alter_topic_options_alter_topic_unique_together_and_more"),
        ("products", "0018_sumoplaceholderpage_singleproductindexpage"),
        ("products", "0019_alter_singleproductindexpage_body"),
        ("products", "0020_remove_sumoplaceholderpage_page_ptr_and_more"),
    ]

    initial = True

    dependencies = []

    operations = [
        migrations.CreateModel(
            name="Platform",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True, primary_key=True, serialize=False, verbose_name="ID"
                    ),
                ),
                ("name", models.CharField(max_length=255)),
                ("slug", models.SlugField()),
                ("visible", models.BooleanField(default=False)),
                ("display_order", models.IntegerField()),
            ],
            options={
                "abstract": False,
            },
        ),
        migrations.CreateModel(
            name="Product",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True, primary_key=True, serialize=False, verbose_name="ID"
                    ),
                ),
                ("title", models.CharField(db_index=True, max_length=255)),
                ("slug", models.SlugField()),
                ("description", models.TextField()),
                (
                    "image",
                    kitsune.sumo.fields.ImagePlusField(
                        blank=True,
                        help_text="Used on the the home page. Must be 484x244.",
                        max_length=250,
                        null=True,
                        upload_to="uploads/products/",
                    ),
                ),
                ("image_offset", models.IntegerField(default=None, editable=False, null=True)),
                (
                    "image_cachebuster",
                    models.CharField(default=None, editable=False, max_length=32, null=True),
                ),
                ("sprite_height", models.IntegerField(default=None, editable=False, null=True)),
                ("display_order", models.IntegerField()),
                ("visible", models.BooleanField(default=False)),
                ("platforms", models.ManyToManyField(to="products.platform")),
                (
                    "image_alternate",
                    kitsune.sumo.fields.ImagePlusField(
                        blank=True,
                        help_text="Used everywhere except the home page. Must be 96x96.",
                        max_length=250,
                        null=True,
                        upload_to="uploads/products/",
                    ),
                ),
                ("codename", models.CharField(blank=True, default="", max_length=255)),
            ],
            options={
                "ordering": ["display_order"],
            },
        ),
        migrations.CreateModel(
            name="Version",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True, primary_key=True, serialize=False, verbose_name="ID"
                    ),
                ),
                ("name", models.CharField(max_length=255)),
                ("slug", models.CharField(db_index=True, max_length=255)),
                ("min_version", models.FloatField()),
                ("max_version", models.FloatField()),
                ("visible", models.BooleanField(default=False)),
                ("default", models.BooleanField(default=False)),
                (
                    "product",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        related_name="versions",
                        to="products.product",
                    ),
                ),
            ],
            options={
                "ordering": ["-max_version"],
            },
        ),
        migrations.CreateModel(
            name="Topic",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True, primary_key=True, serialize=False, verbose_name="ID"
                    ),
                ),
                ("title", models.CharField(db_index=True, max_length=255)),
                ("slug", models.CharField(db_index=True, max_length=255)),
                ("description", models.TextField()),
                (
                    "image",
                    kitsune.sumo.fields.ImagePlusField(
                        blank=True, max_length=250, null=True, upload_to="uploads/topics/"
                    ),
                ),
                ("display_order", models.IntegerField()),
                ("visible", models.BooleanField(default=False)),
                (
                    "parent",
                    models.ForeignKey(
                        blank=True,
                        null=True,
                        on_delete=django.db.models.deletion.CASCADE,
                        related_name="subtopics",
                        to="products.topic",
                    ),
                ),
                (
                    "product",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        related_name="topics",
                        to="products.product",
                    ),
                ),
                (
                    "in_aaq",
                    models.BooleanField(
                        default=False,
                        help_text="Whether this topic is shown to users in the AAQ or not.",
                    ),
                ),
            ],
            options={
                "ordering": ["product", "display_order"],
            },
        ),
        migrations.CreateModel(
            name="ProductTopic",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True, primary_key=True, serialize=False, verbose_name="ID"
                    ),
                ),
                ("created", models.DateTimeField(auto_now_add=True)),
                (
                    "product",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="products.product"
                    ),
                ),
                (
                    "topic",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE, to="products.topic"
                    ),
                ),
            ],
        ),
        migrations.AddField(
            model_name="topic",
            name="products",
            field=models.ManyToManyField(
                related_name="m2m_topics", through="products.ProductTopic", to="products.product"
            ),
        ),
        migrations.AddConstraint(
            model_name="producttopic",
            constraint=models.UniqueConstraint(
                fields=("product", "topic"), name="unique_product_topic"
            ),
        ),
        migrations.RunPython(
            code=copy_product_to_products,
            reverse_code=copy_product_to_products_backwards,
        ),
        migrations.CreateModel(
            name="TopicSlugHistory",
            fields=[
                (
                    "id",
                    models.AutoField(
                        auto_created=True, primary_key=True, serialize=False, verbose_name="ID"
                    ),
                ),
                ("slug", models.SlugField(max_length=255, unique=True)),
                ("created", models.DateTimeField(auto_now_add=True)),
                (
                    "topic",
                    models.ForeignKey(
                        on_delete=django.db.models.deletion.CASCADE,
                        related_name="slug_history",
                        to="products.topic",
                    ),
                ),
            ],
            options={
                "verbose_name_plural": "Topic slug history",
                "ordering": ["-created"],
            },
        ),
        migrations.RunPython(
            code=move_kb_articles,
            reverse_code=move_kb_articles_backwards,
        ),
        migrations.AddField(
            model_name="product",
            name="is_archived",
            field=models.BooleanField(default=False),
        ),
        migrations.AddField(
            model_name="topic",
            name="is_archived",
            field=models.BooleanField(default=False),
        ),
        migrations.RunPython(
            code=archive_old_topics,
            reverse_code=archive_old_topics_backwards,
        ),
        migrations.AlterField(
            model_name="topic",
            name="product",
            field=models.ForeignKey(
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                related_name="topics",
                to="products.product",
            ),
        ),
        migrations.RunPython(
            code=create_new_taxonomy_topics,
            reverse_code=create_new_taxonomy_topics_backwards,
        ),
        migrations.RunPython(
            code=migrate_topics,
            reverse_code=migrate_topics_backwards,
        ),
        migrations.AddField(
            model_name="topic",
            name="in_nav",
            field=models.BooleanField(
                default=False, help_text="Whether this topic is shown in navigation menus."
            ),
        ),
        migrations.AlterField(
            model_name="topic",
            name="product",
            field=models.ForeignKey(
                blank=True,
                null=True,
                on_delete=django.db.models.deletion.CASCADE,
                related_name="topics",
                to="products.product",
            ),
        ),
        migrations.AlterModelOptions(
            name="topic",
            options={"ordering": ["title", "display_order"]},
        ),
        migrations.AlterUniqueTogether(
            name="topic",
            unique_together=set(),
        ),
        migrations.RemoveField(
            model_name="topic",
            name="product",
        ),
    ]
