import json
import logging
from datetime import date, timedelta

from django.conf import settings
from django.http import Http404, HttpResponse, HttpResponseRedirect
from django.shortcuts import get_object_or_404, render
from django.utils.translation import gettext as _
from django.views.decorators.http import require_GET

from kitsune.dashboards import PERIODS
from kitsune.dashboards.readouts import (
    CONTRIBUTOR_READOUTS,
    L10N_READOUTS,
    READOUTS,
    kb_overview_rows,
    l10n_overview_rows,
)
from kitsune.dashboards.utils import get_locales_by_visit, render_readouts
from kitsune.products.models import Product
from kitsune.sumo.urlresolvers import reverse
from kitsune.sumo.utils import smart_int
from kitsune.wiki.config import CATEGORIES

log = logging.getLogger("k.dashboards")


def _kb_readout(request, readout_slug, readouts, locale=None, mode=None, product=None):
    """Instantiate and return the readout with the given slug.

    Raise Http404 if there is no such readout.

    """
    if readout_slug not in readouts:
        raise Http404
    return readouts[readout_slug](request, locale=locale, mode=mode, product=product)


def _kb_detail(
    request, readout_slug, readouts, main_view_name, main_dash_title, locale=None, product=None
):
    """Show all the rows for the given KB article statistics table."""
    return render(
        request,
        "dashboards/kb_detail.html",
        {
            "readout": _kb_readout(request, readout_slug, readouts, locale, product=product),
            "locale": locale,
            "main_dash_view": main_view_name,
            "main_dash_title": main_dash_title,
            "product": product,
            "products": Product.objects.filter(visible=True),
        },
    )


@require_GET
def contributors_detail(request, readout_slug):
    """Show all the rows for the given contributor dashboard table."""
    product = _get_product(request)

    return _kb_detail(
        request,
        readout_slug,
        CONTRIBUTOR_READOUTS,
        "dashboards.contributors",
        _("Knowledge Base Dashboard"),
        locale=settings.WIKI_DEFAULT_LANGUAGE,
        product=product,
    )


@require_GET
def contributors_overview(request):
    product = _get_product(request)
    category = _get_category(request)

    return render(
        request,
        "dashboards/contributors_overview.html",
        {
            "overview_rows": kb_overview_rows(
                user=request.user,
                locale=request.LANGUAGE_CODE,
                product=product,
                mode=smart_int(request.GET.get("mode"), None),
                max=None,
                category=category,
            ),
            "main_dash_view": "dashboards.contributors",
            "main_dash_title": _("Knowledge Base Dashboard"),
            "locale": request.LANGUAGE_CODE,
            "product": product,
            "products": Product.objects.filter(visible=True),
            "category": category,
            "categories": CATEGORIES,
        },
    )


@require_GET
def localization_detail(request, readout_slug):
    """Show all the rows for the given localizer dashboard table."""
    product = _get_product(request)

    return _kb_detail(
        request,
        readout_slug,
        L10N_READOUTS,
        "dashboards.localization",
        _("Localization Dashboard"),
        product=product,
    )


@require_GET
def localization(request):
    """Render aggregate data about articles in a non-default locale."""
    if request.LANGUAGE_CODE == settings.WIKI_DEFAULT_LANGUAGE:
        return HttpResponseRedirect(reverse("dashboards.contributors"))

    product = _get_product(request)

    data = {
        "overview_rows": l10n_overview_rows(request.LANGUAGE_CODE, product=product),
    }

    return render_readouts(
        request, L10N_READOUTS, "localization.html", extra_data=data, product=product
    )


@require_GET
def contributors(request):
    """Render aggregate data about the articles in the default locale."""
    product = _get_product(request)
    category = _get_category(request)

    return render_readouts(
        request,
        CONTRIBUTOR_READOUTS,
        "contributors.html",
        locale=settings.WIKI_DEFAULT_LANGUAGE,
        product=product,
        extra_data={
            "overview_rows": kb_overview_rows(
                user=request.user,
                locale=request.LANGUAGE_CODE,
                product=product,
                mode=smart_int(request.GET.get("mode"), None),
                max=smart_int(request.GET.get("max"), 10),
                category=category,
            ),
            "overview_modes": PERIODS,
            "category": category,
            "categories": CATEGORIES,
            "locale": request.LANGUAGE_CODE,
        },
    )


@require_GET
def contributors_old(request):
    """Render aggregate data about the articles in the default locale."""
    product = _get_product(request)

    return render_readouts(
        request,
        CONTRIBUTOR_READOUTS,
        "contributors_old.html",
        locale=settings.WIKI_DEFAULT_LANGUAGE,
        product=product,
    )


@require_GET
def wiki_rows(request, readout_slug):
    """Return the table contents HTML for the given readout and mode."""
    product = _get_product(request)

    readout = _kb_readout(
        request,
        readout_slug,
        READOUTS,
        locale=request.GET.get("locale"),
        mode=smart_int(request.GET.get("mode"), None),
        product=product,
    )
    max_rows = smart_int(request.GET.get("max"), fallback=None)
    return HttpResponse(readout.render(max_rows=max_rows))


@require_GET
def contributors_overview_rows(request):
    product = _get_product(request)

    overview_rows = kb_overview_rows(
        user=request.user,
        locale=request.LANGUAGE_CODE,
        product=product,
        mode=smart_int(request.GET.get("mode"), None),
        max=smart_int(request.GET.get("max"), 10),
    )

    return render(
        request, "dashboards/includes/kb_overview.html", {"overview_rows": overview_rows}
    )


@require_GET
def locale_metrics(request, locale_code):
    """The kb metrics dashboard for a specific locale."""

    if locale_code not in settings.SUMO_LANGUAGES:
        raise Http404

    product = _get_product(request)

    return render(
        request,
        "dashboards/locale_metrics.html",
        {
            "current_locale": locale_code,
            "product": product,
            "products": Product.objects.filter(visible=True),
        },
    )


@require_GET
def aggregated_metrics(request):
    """The aggregated (all locales) kb metrics dashboard."""
    today = date.today()
    locales = get_locales_by_visit(today - timedelta(days=30), today)
    product = _get_product(request)

    return render(
        request,
        "dashboards/aggregated_metrics.html",
        {
            "locales_json": json.dumps(settings.SUMO_LANGUAGES),
            "locales": locales,
            "product": product,
            "products": Product.objects.filter(visible=True),
        },
    )


def _get_product(request):
    product_slug = request.GET.get("product")
    if product_slug:
        return get_object_or_404(Product, slug=product_slug)

    return None


def _get_category(request):
    category = request.GET.get("category")

    if category:
        for c in CATEGORIES:
            if str(c[0]) == category:
                return c[0]
        raise Http404("Invalid category.")

    return None
