import logging
from collections import OrderedDict

from django.conf import settings
from django.core.cache import cache
from django.shortcuts import render
from google.auth.exceptions import GoogleAuthError
from OpenSSL.crypto import Error as OpenSSLError

from kitsune.lib.sumo_locales import LOCALES
from kitsune.products.models import Product
from kitsune.sumo.googleanalytics import visitors_by_locale
from kitsune.wiki.events import (
    ApproveRevisionInLocaleEvent,
    ReadyRevisionEvent,
    ReviewableRevisionInLocaleEvent,
)

log = logging.getLogger("k.dashboards")


def render_readouts(request, readouts, template, locale=None, extra_data=None, product=None):
    """Render a readouts, possibly with overview page.

    Use the given template, pass the template the given readouts, limit the
    considered data to the given locale, and pass along anything in the
    `extra_data` dict to the template in addition to the standard data.

    """
    current_locale = locale or request.LANGUAGE_CODE
    on_default_locale = request.LANGUAGE_CODE == settings.WIKI_DEFAULT_LANGUAGE

    default_kwargs = {
        "locale": settings.WIKI_DEFAULT_LANGUAGE,
    }
    locale_kwargs = {
        "locale": request.LANGUAGE_CODE,
    }
    ready_kwargs = {}

    if product is not None:
        default_kwargs["product"] = product.slug
        locale_kwargs["product"] = product.slug
        ready_kwargs["product"] = product.slug

    data = {
        "readouts": OrderedDict(
            (slug, class_(request, locale=locale, product=product))
            for slug, class_ in readouts.items()
            if class_.should_show_to(request)
        ),
        "default_locale": settings.WIKI_DEFAULT_LANGUAGE,
        "default_locale_name": LOCALES[settings.WIKI_DEFAULT_LANGUAGE].native,
        "current_locale": current_locale,
        "current_locale_name": LOCALES[current_locale].native,
        "request_locale_name": LOCALES[request.LANGUAGE_CODE].native,
        "is_watching_default_approved": ApproveRevisionInLocaleEvent.is_notifying(
            request.user, **default_kwargs
        ),
        "is_watching_other_approved": (
            None
            if on_default_locale
            else ApproveRevisionInLocaleEvent.is_notifying(request.user, **locale_kwargs)
        ),
        "is_watching_default_locale": (
            ReviewableRevisionInLocaleEvent.is_notifying(request.user, **default_kwargs)
        ),
        "is_watching_other_locale": (
            None
            if on_default_locale
            else ReviewableRevisionInLocaleEvent.is_notifying(request.user, **locale_kwargs)
        ),
        "is_watching_default_ready": ReadyRevisionEvent.is_notifying(request.user, **ready_kwargs),
        "on_default_locale": on_default_locale,
        "product": product,
        "products": Product.objects.filter(visible=True),
    }
    if extra_data:
        data.update(extra_data)
    return render(request, "dashboards/" + template, data)


def get_locales_by_visit(start_date, end_date):
    """Get a list of (locale, visits) tuples sorted descending by visits."""

    cache_key = "locales_sorted_by_visits:{start}:{end}".format(start=start_date, end=end_date)

    sorted_locales = cache.get(cache_key)
    if sorted_locales is None:
        try:
            sorted_locales = sorted(
                visitors_by_locale(start_date, end_date).items(), key=lambda x: x[1], reverse=True
            )
            cache.add(cache_key, sorted_locales, settings.CACHE_LONG_TIMEOUT)
        except (GoogleAuthError, OpenSSLError):
            # Just return all locales with 0s for visits.
            log.exception(
                "Something went wrong getting visitors by locale "
                "from Google Analytics. Nobody got a 500 though."
            )
            sorted_locales = [(lang, 0) for lang in settings.SUMO_LANGUAGES]

    return sorted_locales
