"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.lexCommentString = void 0;
const lexer_1 = require("./lexer");
function* lexCommentString(file) {
    // Wrapper around our real lex function to collapse adjacent text tokens.
    let textToken;
    for (const token of lexCommentString2(file)) {
        if (token.kind === lexer_1.TokenSyntaxKind.Text) {
            if (textToken) {
                textToken.text += token.text;
            }
            else {
                textToken = token;
            }
        }
        else {
            if (textToken) {
                yield textToken;
                textToken = void 0;
            }
            yield token;
        }
    }
    if (textToken) {
        yield textToken;
    }
    return;
}
exports.lexCommentString = lexCommentString;
function* lexCommentString2(file) {
    let pos = 0;
    let end = file.length;
    // Skip leading whitespace
    while (pos < end && /\s/.test(file[pos])) {
        pos++;
    }
    // Trailing whitespace
    while (pos < end && /\s/.test(file[end - 1])) {
        end--;
    }
    let lineStart = true;
    let braceStartsType = false;
    for (;;) {
        if (pos >= end) {
            return;
        }
        if (lineStart) {
            lineStart = false;
        }
        switch (file[pos]) {
            case "\n":
                yield makeToken(lexer_1.TokenSyntaxKind.NewLine, 1);
                lineStart = true;
                break;
            case "{":
                if (braceStartsType && nextNonWs(pos + 1) !== "@") {
                    yield makeToken(lexer_1.TokenSyntaxKind.TypeAnnotation, findEndOfType(pos) - pos);
                    braceStartsType = false;
                }
                else {
                    yield makeToken(lexer_1.TokenSyntaxKind.OpenBrace, 1);
                }
                break;
            case "}":
                yield makeToken(lexer_1.TokenSyntaxKind.CloseBrace, 1);
                braceStartsType = false;
                break;
            case "`": {
                // Markdown's code rules are a royal pain. This could be one of several things.
                // 1. Inline code: <1-n ticks><text><same number of ticks>
                // 2. Code block: <3 ticks><language, no ticks>\n<text>\n<3 ticks>\n
                // 3. Unmatched tick(s), not code, but part of some text.
                // We don't quite handle #2 correctly yet. PR welcome!
                braceStartsType = false;
                let tickCount = 1;
                let lookahead = pos;
                while (lookahead + 1 < end && file[lookahead + 1] === "`") {
                    tickCount++;
                    lookahead++;
                }
                let lookaheadStart = pos;
                const codeText = [];
                lookahead++;
                while (lookahead < end) {
                    if (lookaheadExactlyNTicks(lookahead, tickCount)) {
                        lookahead += tickCount;
                        codeText.push(file.substring(lookaheadStart, lookahead));
                        yield {
                            kind: lexer_1.TokenSyntaxKind.Code,
                            text: codeText.join(""),
                            pos,
                        };
                        pos = lookahead;
                        break;
                    }
                    else if (file[lookahead] === "`") {
                        while (lookahead < end && file[lookahead] === "`") {
                            lookahead++;
                        }
                    }
                    else if (file[lookahead] === "\\" &&
                        lookahead + 1 < end &&
                        file[lookahead + 1] !== "\n") {
                        lookahead += 2;
                    }
                    else if (file[lookahead] === "\n") {
                        lookahead++;
                        codeText.push(file.substring(lookaheadStart, lookahead));
                        lookaheadStart = lookahead;
                    }
                    else {
                        lookahead++;
                    }
                }
                if (lookahead >= end && pos !== lookahead) {
                    if (tickCount === 3 &&
                        file.substring(pos, end).includes("\n")) {
                        codeText.push(file.substring(lookaheadStart, end));
                        yield {
                            kind: lexer_1.TokenSyntaxKind.Code,
                            text: codeText.join(""),
                            pos,
                        };
                        pos = lookahead;
                    }
                    else {
                        yield makeToken(lexer_1.TokenSyntaxKind.Text, tickCount);
                    }
                }
                break;
            }
            case "@": {
                let lookahead = pos + 1;
                while (lookahead < end && /[a-z]/i.test(file[lookahead])) {
                    lookahead++;
                }
                if (lookahead !== pos + 1) {
                    while (lookahead < end &&
                        /[a-z0-9]/i.test(file[lookahead])) {
                        lookahead++;
                    }
                }
                if (lookahead !== pos + 1 &&
                    (lookahead === end || /[\s}]/.test(file[lookahead]))) {
                    braceStartsType = true;
                    yield makeToken(lexer_1.TokenSyntaxKind.Tag, lookahead - pos);
                    break;
                }
            }
            // fall through if we didn't find something that looks like a tag
            default: {
                const textParts = [];
                let lookaheadStart = pos;
                let lookahead = pos;
                while (lookahead < end) {
                    if ("{}\n`".includes(file[lookahead]))
                        break;
                    if (lookahead !== pos &&
                        file[lookahead] === "@" &&
                        /\s/.test(file[lookahead - 1])) {
                        // Probably the start of a modifier tag
                        break;
                    }
                    if (file[lookahead] === "\\" &&
                        lookahead + 1 < end &&
                        "{}@`".includes(file[lookahead + 1])) {
                        textParts.push(file.substring(lookaheadStart, lookahead), file[lookahead + 1]);
                        lookahead++;
                        lookaheadStart = lookahead + 1;
                    }
                    lookahead++;
                }
                textParts.push(file.substring(lookaheadStart, lookahead));
                if (textParts.some((part) => /\S/.test(part))) {
                    braceStartsType = false;
                }
                // This piece of text had line continuations or escaped text
                yield {
                    kind: lexer_1.TokenSyntaxKind.Text,
                    text: textParts.join(""),
                    pos,
                };
                pos = lookahead;
                break;
            }
        }
    }
    function makeToken(kind, size) {
        const start = pos;
        pos += size;
        return {
            kind,
            text: file.substring(start, pos),
            pos: start,
        };
    }
    function lookaheadExactlyNTicks(pos, n) {
        if (pos + n > end) {
            return false;
        }
        return file.startsWith("`".repeat(n), pos) && file[pos + n] !== "`";
    }
    function findEndOfType(pos) {
        let openBraces = 0;
        while (pos < end) {
            if (file[pos] === "{") {
                openBraces++;
            }
            else if (file[pos] === "}") {
                if (--openBraces === 0) {
                    break;
                }
            }
            else if ("`'\"".includes(file[pos])) {
                pos = findEndOfString(pos);
            }
            pos++;
        }
        if (pos < end && file[pos] === "}") {
            pos++;
        }
        return pos;
    }
    function findEndOfString(pos) {
        const endOfString = file[pos];
        pos++;
        while (pos < end) {
            if (file[pos] === endOfString) {
                break;
            }
            else if (file[pos] === "\\") {
                pos++; // Skip escaped character
            }
            else if (endOfString === "`" &&
                file[pos] === "$" &&
                file[pos + 1] === "{") {
                // Template literal with data inside a ${}
                while (pos < end && file[pos] !== "}") {
                    if ("`'\"".includes(file[pos])) {
                        pos = findEndOfString(pos) + 1;
                    }
                    else {
                        pos++;
                    }
                }
            }
            pos++;
        }
        return pos;
    }
    function nextNonWs(pos) {
        while (pos < end && /\s/.test(file[pos])) {
            pos++;
        }
        return file[pos];
    }
}
