"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.mdxLoader = void 0;
const tslib_1 = require("tslib");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const logger_1 = tslib_1.__importDefault(require("@docusaurus/logger"));
const utils_1 = require("@docusaurus/utils");
const mdx_1 = require("@mdx-js/mdx");
const remark_emoji_1 = tslib_1.__importDefault(require("remark-emoji"));
const stringify_object_1 = tslib_1.__importDefault(require("stringify-object"));
const headings_1 = tslib_1.__importDefault(require("./remark/headings"));
const toc_1 = tslib_1.__importDefault(require("./remark/toc"));
const unwrapMdxCodeBlocks_1 = tslib_1.__importDefault(require("./remark/unwrapMdxCodeBlocks"));
const transformImage_1 = tslib_1.__importDefault(require("./remark/transformImage"));
const transformLinks_1 = tslib_1.__importDefault(require("./remark/transformLinks"));
const mermaid_1 = tslib_1.__importDefault(require("./remark/mermaid"));
const admonitions_1 = tslib_1.__importDefault(require("./remark/admonitions"));
const { loaders: { inlineMarkdownImageFileLoader }, } = (0, utils_1.getFileLoaderUtils)();
const pragma = `
/* @jsxRuntime classic */
/* @jsx mdx */
/* @jsxFrag React.Fragment */
`;
const DEFAULT_OPTIONS = {
    admonitions: true,
    rehypePlugins: [],
    remarkPlugins: [unwrapMdxCodeBlocks_1.default, remark_emoji_1.default, headings_1.default, toc_1.default],
    beforeDefaultRemarkPlugins: [],
    beforeDefaultRehypePlugins: [],
};
const compilerCache = new Map();
/**
 * When this throws, it generally means that there's no metadata file associated
 * with this MDX document. It can happen when using MDX partials (usually
 * starting with _). That's why it's important to provide the `isMDXPartial`
 * function in config
 */
async function readMetadataPath(metadataPath) {
    try {
        return await fs_extra_1.default.readFile(metadataPath, 'utf8');
    }
    catch (err) {
        logger_1.default.error `MDX loader can't read MDX metadata file path=${metadataPath}. Maybe the isMDXPartial option function was not provided?`;
        throw err;
    }
}
/**
 * Converts assets an object with Webpack require calls code.
 * This is useful for mdx files to reference co-located assets using relative
 * paths. Those assets should enter the Webpack assets pipeline and be hashed.
 * For now, we only handle that for images and paths starting with `./`:
 *
 * `{image: "./myImage.png"}` => `{image: require("./myImage.png")}`
 */
function createAssetsExportCode(assets) {
    if (typeof assets !== 'object' ||
        !assets ||
        Object.keys(assets).length === 0) {
        return 'undefined';
    }
    // TODO implementation can be completed/enhanced
    function createAssetValueCode(assetValue) {
        if (Array.isArray(assetValue)) {
            const arrayItemCodes = assetValue.map((item) => createAssetValueCode(item) ?? 'undefined');
            return `[${arrayItemCodes.join(', ')}]`;
        }
        // Only process string values starting with ./
        // We could enhance this logic and check if file exists on disc?
        if (typeof assetValue === 'string' && assetValue.startsWith('./')) {
            // TODO do we have other use-cases than image assets?
            // Probably not worth adding more support, as we want to move to Webpack 5 new asset system (https://github.com/facebook/docusaurus/pull/4708)
            const inlineLoader = inlineMarkdownImageFileLoader;
            return `require("${inlineLoader}${(0, utils_1.escapePath)(assetValue)}").default`;
        }
        return undefined;
    }
    const assetEntries = Object.entries(assets);
    const codeLines = assetEntries
        .map(([key, value]) => {
        const assetRequireCode = createAssetValueCode(value);
        return assetRequireCode ? `"${key}": ${assetRequireCode},` : undefined;
    })
        .filter(Boolean);
    return `{\n${codeLines.join('\n')}\n}`;
}
function getAdmonitionsPlugins(admonitionsOption) {
    if (admonitionsOption) {
        const plugin = admonitionsOption === true
            ? admonitions_1.default
            : [admonitions_1.default, admonitionsOption];
        return [plugin];
    }
    return [];
}
async function mdxLoader(fileString) {
    const callback = this.async();
    const filePath = this.resourcePath;
    const reqOptions = this.getOptions();
    const { frontMatter, content: contentWithTitle } = (0, utils_1.parseFrontMatter)(fileString);
    const { content, contentTitle } = (0, utils_1.parseMarkdownContentTitle)(contentWithTitle, {
        removeContentTitle: reqOptions.removeContentTitle,
    });
    const hasFrontMatter = Object.keys(frontMatter).length > 0;
    if (!compilerCache.has(this.query)) {
        const remarkPlugins = [
            ...(reqOptions.beforeDefaultRemarkPlugins ?? []),
            ...getAdmonitionsPlugins(reqOptions.admonitions ?? false),
            ...DEFAULT_OPTIONS.remarkPlugins,
            ...(reqOptions.markdownConfig.mermaid ? [mermaid_1.default] : []),
            [
                transformImage_1.default,
                {
                    staticDirs: reqOptions.staticDirs,
                    siteDir: reqOptions.siteDir,
                },
            ],
            [
                transformLinks_1.default,
                {
                    staticDirs: reqOptions.staticDirs,
                    siteDir: reqOptions.siteDir,
                },
            ],
            ...(reqOptions.remarkPlugins ?? []),
        ];
        const rehypePlugins = [
            ...(reqOptions.beforeDefaultRehypePlugins ?? []),
            ...DEFAULT_OPTIONS.rehypePlugins,
            ...(reqOptions.rehypePlugins ?? []),
        ];
        const options = {
            ...reqOptions,
            remarkPlugins,
            rehypePlugins,
        };
        compilerCache.set(this.query, [(0, mdx_1.createCompiler)(options), options]);
    }
    const [compiler, options] = compilerCache.get(this.query);
    let result;
    try {
        result = await compiler
            .process({
            contents: content,
            path: this.resourcePath,
        })
            .then((res) => res.toString());
    }
    catch (err) {
        return callback(err);
    }
    // MDX partials are MDX files starting with _ or in a folder starting with _
    // Partial are not expected to have associated metadata files or front matter
    const isMDXPartial = options.isMDXPartial?.(filePath);
    if (isMDXPartial && hasFrontMatter) {
        const errorMessage = `Docusaurus MDX partial files should not contain front matter.
Those partial files use the _ prefix as a convention by default, but this is configurable.
File at ${filePath} contains front matter that will be ignored:
${JSON.stringify(frontMatter, null, 2)}`;
        if (!options.isMDXPartialFrontMatterWarningDisabled) {
            const shouldError = process.env.NODE_ENV === 'test' || process.env.CI;
            if (shouldError) {
                return callback(new Error(errorMessage));
            }
            logger_1.default.warn(errorMessage);
        }
    }
    function getMetadataPath() {
        if (!isMDXPartial) {
            // Read metadata for this MDX and export it.
            if (options.metadataPath && typeof options.metadataPath === 'function') {
                return options.metadataPath(filePath);
            }
        }
        return undefined;
    }
    const metadataPath = getMetadataPath();
    if (metadataPath) {
        this.addDependency(metadataPath);
    }
    const metadataJsonString = metadataPath
        ? await readMetadataPath(metadataPath)
        : undefined;
    const metadata = metadataJsonString
        ? JSON.parse(metadataJsonString)
        : undefined;
    const assets = reqOptions.createAssets && metadata
        ? reqOptions.createAssets({ frontMatter, metadata })
        : undefined;
    const exportsCode = `
export const frontMatter = ${(0, stringify_object_1.default)(frontMatter)};
export const contentTitle = ${(0, stringify_object_1.default)(contentTitle)};
${metadataJsonString ? `export const metadata = ${metadataJsonString};` : ''}
${assets ? `export const assets = ${createAssetsExportCode(assets)};` : ''}
`;
    const code = `
${pragma}
import React from 'react';
import { mdx } from '@mdx-js/react';

${exportsCode}
${result}
`;
    return callback(null, code);
}
exports.mdxLoader = mdxLoader;
//# sourceMappingURL=loader.js.map