"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addTrailingPathSeparator = exports.escapePath = exports.aliasedSitePath = exports.toMessageRelativeFilePath = exports.posixPath = exports.shortName = exports.isNameTooLong = void 0;
const tslib_1 = require("tslib");
const path_1 = tslib_1.__importDefault(require("path"));
// Based on https://github.com/gatsbyjs/gatsby/pull/21518/files
// macOS (APFS) and Windows (NTFS) filename length limit = 255 chars,
// Others = 255 bytes
const MAX_PATH_SEGMENT_CHARS = 255;
const MAX_PATH_SEGMENT_BYTES = 255;
// Space for appending things to the string like file extensions and so on
const SPACE_FOR_APPENDING = 10;
const isMacOs = () => process.platform === 'darwin';
const isWindows = () => process.platform === 'win32';
const isNameTooLong = (str) => 
// Not entirely correct: we can't assume FS from OS. But good enough?
isMacOs() || isWindows()
    ? // Windows (NTFS) and macOS (APFS) filename length limit (255 chars)
        str.length + SPACE_FOR_APPENDING > MAX_PATH_SEGMENT_CHARS
    : // Other (255 bytes)
        Buffer.from(str).length + SPACE_FOR_APPENDING > MAX_PATH_SEGMENT_BYTES;
exports.isNameTooLong = isNameTooLong;
function shortName(str) {
    if (isMacOs() || isWindows()) {
        const overflowingChars = str.length - MAX_PATH_SEGMENT_CHARS;
        return str.slice(0, str.length - overflowingChars - SPACE_FOR_APPENDING - 1);
    }
    const strBuffer = Buffer.from(str);
    const overflowingBytes = Buffer.byteLength(strBuffer) - MAX_PATH_SEGMENT_BYTES;
    return strBuffer
        .slice(0, Buffer.byteLength(strBuffer) - overflowingBytes - SPACE_FOR_APPENDING - 1)
        .toString();
}
exports.shortName = shortName;
/**
 * Convert Windows backslash paths to posix style paths.
 * E.g: endi\lie -> endi/lie
 *
 * Returns original path if the posix counterpart is not valid Windows path.
 * This makes the legacy code that uses posixPath safe; but also makes it less
 * useful when you actually want a path with forward slashes (e.g. for URL)
 *
 * Adopted from https://github.com/sindresorhus/slash/blob/main/index.js
 */
function posixPath(str) {
    const isExtendedLengthPath = str.startsWith('\\\\?\\');
    if (isExtendedLengthPath) {
        return str;
    }
    return str.replace(/\\/g, '/');
}
exports.posixPath = posixPath;
/**
 * When you want to display a path in a message/warning/error, it's more
 * convenient to:
 *
 * - make it relative to `cwd()`
 * - convert to posix (ie not using windows \ path separator)
 *
 * This way, Jest tests can run more reliably on any computer/CI on both
 * Unix/Windows
 * For Windows users this is not perfect (as they see / instead of \) but it's
 * probably good enough
 */
function toMessageRelativeFilePath(filePath) {
    return posixPath(path_1.default.relative(process.cwd(), filePath));
}
exports.toMessageRelativeFilePath = toMessageRelativeFilePath;
/**
 * Alias filepath relative to site directory, very useful so that we
 * don't expose user's site structure.
 * Example: some/path/to/website/docs/foo.md -> @site/docs/foo.md
 */
function aliasedSitePath(filePath, siteDir) {
    const relativePath = posixPath(path_1.default.relative(siteDir, filePath));
    // Cannot use path.join() as it resolves '../' and removes
    // the '@site'. Let webpack loader resolve it.
    return `@site/${relativePath}`;
}
exports.aliasedSitePath = aliasedSitePath;
/**
 * When you have a path like C:\X\Y
 * It is not safe to use directly when generating code
 * For example, this would fail due to unescaped \:
 * `<img src={require('${filePath}')} />`
 * But this would work: `<img src={require('${escapePath(filePath)}')} />`
 *
 * posixPath can't be used in all cases, because forward slashes are only valid
 * Windows paths when they don't contain non-ascii characters, and posixPath
 * doesn't escape those that fail to be converted.
 */
function escapePath(str) {
    const escaped = JSON.stringify(str);
    // Remove the " around the json string;
    return escaped.substring(1, escaped.length - 1);
}
exports.escapePath = escapePath;
function addTrailingPathSeparator(str) {
    return str.endsWith(path_1.default.sep)
        ? str
        : // If this is Windows, we need to change the forward slash to backward
            `${str.replace(/\/$/, '')}${path_1.default.sep}`;
}
exports.addTrailingPathSeparator = addTrailingPathSeparator;
//# sourceMappingURL=pathUtils.js.map