"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.serve = void 0;
const tslib_1 = require("tslib");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const http_1 = tslib_1.__importDefault(require("http"));
const path_1 = tslib_1.__importDefault(require("path"));
const logger_1 = tslib_1.__importDefault(require("@docusaurus/logger"));
const utils_1 = require("@docusaurus/utils");
const serve_handler_1 = tslib_1.__importDefault(require("serve-handler"));
const openBrowser_1 = tslib_1.__importDefault(require("react-dev-utils/openBrowser"));
const config_1 = require("../server/config");
const build_1 = require("./build");
const getHostPort_1 = require("../server/getHostPort");
async function serve(siteDirParam = '.', cliOptions = {}) {
    const siteDir = await fs_extra_1.default.realpath(siteDirParam);
    const buildDir = cliOptions.dir ?? utils_1.DEFAULT_BUILD_DIR_NAME;
    let dir = path_1.default.resolve(siteDir, buildDir);
    if (cliOptions.build) {
        dir = await (0, build_1.build)(siteDir, {
            config: cliOptions.config,
            outDir: dir,
        }, false);
    }
    const { host, port } = await (0, getHostPort_1.getHostPort)(cliOptions);
    if (port === null) {
        process.exit();
    }
    const { siteConfig: { baseUrl, trailingSlash }, } = await (0, config_1.loadSiteConfig)({
        siteDir,
        customConfigFilePath: cliOptions.config,
    });
    const servingUrl = `http://${host}:${port}`;
    const server = http_1.default.createServer((req, res) => {
        // Automatically redirect requests to /baseUrl/
        if (!req.url?.startsWith(baseUrl)) {
            res.writeHead(302, {
                Location: baseUrl,
            });
            res.end();
            return;
        }
        // Remove baseUrl before calling serveHandler, because /baseUrl/ should
        // serve /build/index.html, not /build/baseUrl/index.html (does not exist)
        req.url = req.url.replace(baseUrl, '/');
        (0, serve_handler_1.default)(req, res, {
            cleanUrls: true,
            public: dir,
            trailingSlash,
            directoryListing: false,
        });
    });
    const url = servingUrl + baseUrl;
    logger_1.default.success `Serving path=${buildDir} directory at: url=${url}`;
    server.listen(port);
    if (cliOptions.open && !process.env.CI) {
        (0, openBrowser_1.default)(url);
    }
}
exports.serve = serve;
