"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.processSidebars = void 0;
const tslib_1 = require("tslib");
const lodash_1 = tslib_1.__importDefault(require("lodash"));
const combine_promises_1 = tslib_1.__importDefault(require("combine-promises"));
const generator_1 = require("./generator");
const validation_1 = require("./validation");
const docs_1 = require("../docs");
function toSidebarItemsGeneratorDoc(doc) {
    return lodash_1.default.pick(doc, [
        'id',
        'unversionedId',
        'title',
        'frontMatter',
        'source',
        'sourceDirName',
        'sidebarPosition',
    ]);
}
function toSidebarItemsGeneratorVersion(version) {
    return lodash_1.default.pick(version, ['versionName', 'contentPath']);
}
// Handle the generation of autogenerated sidebar items and other
// post-processing checks
async function processSidebar(unprocessedSidebar, categoriesMetadata, params) {
    const { sidebarItemsGenerator, numberPrefixParser, docs, version } = params;
    // Just a minor lazy transformation optimization
    const getSidebarItemsGeneratorDocsAndVersion = lodash_1.default.memoize(() => ({
        docs: docs.map(toSidebarItemsGeneratorDoc),
        version: toSidebarItemsGeneratorVersion(version),
    }));
    async function processAutoGeneratedItem(item) {
        const generatedItems = await sidebarItemsGenerator({
            item,
            numberPrefixParser,
            defaultSidebarItemsGenerator: generator_1.DefaultSidebarItemsGenerator,
            isCategoryIndex: docs_1.isCategoryIndex,
            ...getSidebarItemsGeneratorDocsAndVersion(),
            categoriesMetadata,
        });
        // Process again... weird but sidebar item generated might generate some
        // auto-generated items?
        // TODO repeatedly process & unwrap autogenerated items until there are no
        // more autogenerated items, or when loop count (e.g. 10) is reached
        return processItems(generatedItems);
    }
    async function processItem(item) {
        if (item.type === 'category') {
            return [
                {
                    ...item,
                    items: (await Promise.all(item.items.map(processItem))).flat(),
                },
            ];
        }
        if (item.type === 'autogenerated') {
            return processAutoGeneratedItem(item);
        }
        return [item];
    }
    async function processItems(items) {
        return (await Promise.all(items.map(processItem))).flat();
    }
    const processedSidebar = await processItems(unprocessedSidebar);
    return processedSidebar;
}
async function processSidebars(unprocessedSidebars, categoriesMetadata, params) {
    const processedSidebars = await (0, combine_promises_1.default)(lodash_1.default.mapValues(unprocessedSidebars, (unprocessedSidebar) => processSidebar(unprocessedSidebar, categoriesMetadata, params)));
    (0, validation_1.validateSidebars)(processedSidebars);
    return processedSidebars;
}
exports.processSidebars = processSidebars;
