"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.remove = void 0;
const path_1 = require("path");
const fs_1 = require("fs");
const util_1 = require("../utils/util");
/**
 * Recursively deletes directories and files.
 * @function remove
 * @memberof fsn/nextra
 * @param path The path to remove
 * @param options The remove options
 */
async function remove(path, options = {}) {
    options.maxBusyTries = typeof options.maxBusyTries === 'undefined' ? 3 : options.maxBusyTries;
    for (let buysTries = 0; buysTries < options.maxBusyTries; buysTries++) {
        try {
            await rimraf(path, options);
            break;
        }
        catch (err) {
            /* istanbul ignore next: Windows */
            if (util_1.isWindows && (err.code === 'EBUSY' || err.code === 'ENOTEMPTY' || err.code === 'EPERM')) {
                await util_1.setTimeoutPromise(buysTries * 100);
                continue;
            }
            /* istanbul ignore else: Hard to test via CI, such as ENOMEM (running the kernel out of memory) */
            if (err.code === 'ENOENT')
                return;
            else
                throw err;
        }
    }
}
exports.remove = remove;
async function rimraf(myPath, options) {
    try {
        const stats = await fs_1.promises.lstat(myPath);
        if (stats.isDirectory())
            return removeDir(myPath, options);
    }
    catch (err) {
        /* istanbul ignore next: Windows */
        if (util_1.isWindows && err.code === 'EPERM')
            return fixWinEPERM(myPath, options);
        throw err;
    }
    try {
        return await fs_1.promises.unlink(myPath);
    }
    catch (er) {
        /* istanbul ignore next: Windows */
        if (er.code === 'EPERM')
            return util_1.isWindows ? fixWinEPERM(myPath, options) : removeDir(myPath, options, er);
        /* istanbul ignore next: Difficult to reproduce */
        if (er.code === 'EISDIR')
            return removeDir(myPath, options, er);
        else
            throw er;
    }
}
/* istanbul ignore next: Windows */
async function fixWinEPERM(myPath, options) {
    await fs_1.promises.chmod(myPath, 0o666);
    return rimraf(myPath, options);
}
async function removeDir(myPath, options, originalEr = null) {
    try {
        return await fs_1.promises.rmdir(myPath);
    }
    catch (err) {
        /* istanbul ignore else: Difficult to reproduce */
        if (['ENOTEMPTY', 'EEXIST', 'EPERM'].includes(err.code))
            return rmkids(myPath, options);
        else if (err.code === 'ENOTDIR')
            throw originalEr;
        else
            throw err;
    }
}
async function rmkids(myPath, options) {
    const files = await fs_1.promises.readdir(myPath);
    await Promise.all(files.map((file) => remove(path_1.join(myPath, file), options)));
    return fs_1.promises.rmdir(myPath);
}
//# sourceMappingURL=remove.js.map