"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.move = void 0;
const path_1 = require("path");
const fs_1 = require("fs");
const util_1 = require("../utils/util");
const remove_1 = require("./remove");
const mkdirs_1 = require("./mkdirs");
const pathExists_1 = require("./pathExists");
const copy_1 = require("./copy");
/**
 * @function move
 * @memberof fsn/nextra
 * @param source The source path of the file
 * @param destination The destination path of the file
 * @param options The options for the move
 */
async function move(source, destination, options = {}) {
    const overwrite = options.overwrite || false;
    if (path_1.resolve(source) === path_1.resolve(destination))
        return fs_1.promises.access(source);
    const myStat = await fs_1.promises.lstat(source);
    if (myStat.isDirectory() && util_1.isSrcKid(source, destination)) {
        throw new Error('FS-NEXTRA: Moving a parent directory into a child will result in an infinite loop.');
    }
    await mkdirs_1.mkdirs(path_1.dirname(destination));
    if (overwrite) {
        await remove_1.remove(destination);
    }
    else if (await pathExists_1.pathExists(destination)) {
        throw new Error('FS-NEXTRA: Destination already exists.');
    }
    try {
        return await fs_1.promises.rename(source, destination);
    }
    catch (err) {
        /* istanbul ignore next: Can't test via CI */
        if (err.code === 'EXDEV') {
            const opts = {
                overwrite,
                errorOnExist: true
            };
            await copy_1.copy(source, destination, opts);
            return remove_1.remove(source);
        }
        /* istanbul ignore next: Hard to produce, such as ENOMEM (Kernel running out of memory). Can't test via CI */
        throw err;
    }
}
exports.move = move;
//# sourceMappingURL=move.js.map