/// <reference types="node" />
import * as fs from 'fs';
import { URL } from 'url';
export { createReadStream, createWriteStream, unwatchFile, watch, watchFile, Dirent, Stats, ReadStream, WriteStream, constants } from 'fs';
/**
 * Valid types for path values in 'fs'.
 */
declare type PathLike = string | Buffer | URL;
export interface MakeDirectoryOptions {
    /**
     * Indicates whether parent folders should be created.
     * @default false
     */
    recursive?: boolean;
    /**
     * A file mode. If a string is passed, it is parsed as an octal integer. If not specified
     * @default 0o777.
     */
    mode?: number;
}
/**
 * [fs.promises] Asynchronously tests a user's permissions for the file specified by path.
 * @param path A path to a file or directory. If a URL is provided, it must use the `file:` protocol.
 * URL support is _experimental_.
 */
export declare function access(path: PathLike, mode?: number): Promise<void>;
/**
 * [fs.promises] Asynchronously copies `src` to `dest`. By default, `dest` is overwritten if it already exists.
 * Node.js makes no guarantees about the atomicity of the copy operation.
 * If an error occurs after the destination file has been opened for writing, Node.js will attempt
 * to remove the destination.
 * @param src A path to the source file.
 * @param dest A path to the destination file.
 * @param flags An optional integer that specifies the behavior of the copy operation. The only
 * supported flag is `fs.constants.COPYFILE_EXCL`, which causes the copy operation to fail if
 * `dest` already exists.
 */
export declare function copyFile(src: PathLike, dest: PathLike, flags?: number): Promise<void>;
/**
 * [fs.promises] Asynchronous open(2) - open and possibly create a file.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param mode A file mode. If a string is passed, it is parsed as an octal integer. If not
 * supplied, defaults to `0o666`.
 */
export declare function open(path: PathLike, flags: string | number, mode?: string | number): Promise<fs.promises.FileHandle>;
/**
 * [fs.promises] Asynchronously reads data from the file referenced by the supplied `FileHandle`.
 * @param handle A `FileHandle`.
 * @param buffer The buffer that the data will be written to.
 * @param offset The offset in the buffer at which to start writing.
 * @param length The number of bytes to read.
 * @param position The offset from the beginning of the file from which data should be read. If
 * `null`, data will be read from the current position.
 */
export declare function read<Uint8Array>(handle: fs.promises.FileHandle, buffer: any, offset?: number | null, length?: number | null, position?: number | null): Promise<{
    bytesRead: number;
    buffer: Uint8Array;
}>;
/**
 * [fs.promises] Asynchronously writes `buffer` to the file referenced by the supplied `FileHandle`.
 * It is unsafe to call `fsPromises.write()` multiple times on the same file without waiting for the `Promise`
 * to be resolved (or rejected). For this scenario, `fs.createWriteStream` is strongly recommended.
 * @param handle A `FileHandle`.
 * @param buffer The buffer that the data will be written to.
 * @param offset The part of the buffer to be written. If not supplied, defaults to `0`.
 * @param length The number of bytes to write. If not supplied, defaults to `buffer.length - offset`.
 * @param position The offset from the beginning of the file where this data should be written. If not supplied, defaults to the current position.
 */
export declare function write<TBuffer extends Buffer | Uint8Array>(handle: fs.promises.FileHandle, buffer: TBuffer, offset?: number | null, length?: number | null, position?: number | null): Promise<{
    bytesWritten: number;
    buffer: TBuffer;
}>;
/**
 * [fs.promises] Asynchronously writes `string` to the file referenced by the supplied `FileHandle`.
 * It is unsafe to call `fsPromises.write()` multiple times on the same file without waiting for the `Promise`
 * to be resolved (or rejected). For this scenario, `fs.createWriteStream` is strongly recommended.
 * @param handle A `FileHandle`.
 * @param data A string to write. If something other than a string is supplied it will be coerced to a string.
 * @param position The offset from the beginning of the file where this data should be written. If not supplied, defaults to the current position.
 * @param encoding The expected string encoding.
 */
export declare function write(handle: fs.promises.FileHandle, data: any, position?: number | null, encoding?: string | null): Promise<{
    bytesWritten: number;
    buffer: string;
}>;
/**
 * [fs.promises] Asynchronous rename(2) - Change the name or location of a file or directory.
 * @param oldPath A path to a file. If a URL is provided, it must use the `file:` protocol.
 * URL support is _experimental_.
 * @param newPath A path to a file. If a URL is provided, it must use the `file:` protocol.
 * URL support is _experimental_.
 */
export declare function rename(oldPath: PathLike, newPath: PathLike): Promise<void>;
/**
 * [fs.promises] Asynchronous truncate(2) - Truncate a file to a specified length.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param len If not specified, defaults to `0`.
 */
export declare function truncate(path: PathLike, len?: number): Promise<void>;
/**
 * [fs.promises] Asynchronous ftruncate(2) - Truncate a file to a specified length.
 * @param handle A `FileHandle`.
 * @param len If not specified, defaults to `0`.
 */
export declare function ftruncate(handle: fs.promises.FileHandle, len?: number): Promise<void>;
/**
 * [fs.promises] Asynchronous rmdir(2) - delete a directory.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 */
export declare function rmdir(path: PathLike): Promise<void>;
/**
 * Asynchronous fdatasync(2) - synchronize a file's in-core state with storage device.
 * @param handle A `FileHandle`.
 */
export declare function fdatasync(handle: fs.promises.FileHandle): Promise<void>;
/**
 * [fs.promises] Asynchronous fsync(2) - synchronize a file's in-core state with the underlying storage device.
 * @param handle A `FileHandle`.
 */
export declare function fsync(handle: fs.promises.FileHandle): Promise<void>;
/**
 * [fs.promises] Asynchronous mkdir(2) - create a directory.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param options Either the file mode, or an object optionally specifying the file mode and whether parent folders
 * should be created. If a string is passed, it is parsed as an octal integer. If not specified, defaults to `0o777`.
 */
export declare function mkdir(path: PathLike, options?: number | string | MakeDirectoryOptions | null): Promise<void>;
/**
 * [fs.promises] Asynchronous readdir(3) - read a directory.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param options The encoding (or an object specifying the encoding), used as the encoding of the result. If not provided, `'utf8'` is used.
 */
export declare function readdir(path: PathLike, options?: {
    encoding?: BufferEncoding | null;
} | BufferEncoding | null): Promise<string[]>;
export declare function readdir(path: PathLike, options: {
    encoding: 'buffer';
} | 'buffer'): Promise<Buffer[]>;
/**
 * [fs.promises] Asynchronous readlink(2) - read value of a symbolic link.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param options The encoding (or an object specifying the encoding), used as the encoding of the result. If not provided, `'utf8'` is used.
 */
export declare function readlink(path: PathLike, options?: {
    encoding?: BufferEncoding | null;
} | BufferEncoding | null): Promise<string>;
export declare function readlink(path: PathLike, options: {
    encoding: 'buffer';
} | 'buffer'): Promise<Buffer>;
/**
 * [fs.promises] Asynchronous symlink(2) - Create a new symbolic link to an existing file.
 * @param target A path to an existing file. If a URL is provided, it must use the `file:` protocol.
 * @param path A path to the new symlink. If a URL is provided, it must use the `file:` protocol.
 * @param type May be set to `'dir'`, `'file'`, or `'junction'` (default is `'file'`) and is only available on Windows (ignored on other platforms).
 * When using `'junction'`, the `target` argument will automatically be normalized to an absolute path.
 */
export declare function symlink(target: PathLike, path: PathLike, type?: string | null): Promise<void>;
/**
 * [fs.promises] Asynchronous fstat(2) - Get file status.
 * @param handle A `FileHandle`.
 */
export declare function fstat(handle: fs.promises.FileHandle): Promise<fs.Stats>;
/**
 * [fs.promises] Asynchronous lstat(2) - Get file status. Does not dereference symbolic links.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 */
export declare function lstat(path: PathLike): Promise<fs.Stats>;
/**
 * [fs.promises] Asynchronous stat(2) - Get file status.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 */
export declare function stat(path: PathLike): Promise<fs.Stats>;
/**
 * [fs.promises] Asynchronous link(2) - Create a new link (also known as a hard link) to an existing file.
 * @param existingPath A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param newPath A path to a file. If a URL is provided, it must use the `file:` protocol.
 */
export declare function link(existingPath: PathLike, newPath: PathLike): Promise<void>;
/**
 * [fs.promises] Asynchronous unlink(2) - delete a name and possibly the file it refers to.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 */
export declare function unlink(path: PathLike): Promise<void>;
/**
 * [fs.promises] Asynchronous fchmod(2) - Change permissions of a file.
 * @param handle A `FileHandle`.
 * @param mode A file mode. If a string is passed, it is parsed as an octal integer.
 */
export declare function fchmod(handle: fs.promises.FileHandle, mode: string | number): Promise<void>;
/**
 * [fs.promises] Asynchronous chmod(2) - Change permissions of a file.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param mode A file mode. If a string is passed, it is parsed as an octal integer.
 */
export declare function chmod(path: PathLike, mode: string | number): Promise<void>;
/**
 * [fs.promises] Asynchronous lchmod(2) - Change permissions of a file. Does not dereference symbolic links.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param mode A file mode. If a string is passed, it is parsed as an octal integer.
 */
export declare function lchmod(path: PathLike, mode: string | number): Promise<void>;
/**
 * [fs.promises] Asynchronous lchown(2) - Change ownership of a file. Does not dereference symbolic links.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 */
export declare function lchown(path: PathLike, uid: number, gid: number): Promise<void>;
/**
 * [fs.promises] Asynchronous fchown(2) - Change ownership of a file.
 * @param handle A `FileHandle`.
 */
export declare function fchown(handle: fs.promises.FileHandle, uid: number, gid: number): Promise<void>;
/**
 * [fs.promises] Asynchronous chown(2) - Change ownership of a file.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 */
export declare function chown(path: PathLike, uid: number, gid: number): Promise<void>;
/**
 * [fs.promises] Asynchronously change file timestamps of the file referenced by the supplied path.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param atime The last access time. If a string is provided, it will be coerced to number.
 * @param mtime The last modified time. If a string is provided, it will be coerced to number.
 */
export declare function utimes(path: PathLike, atime: string | number | Date, mtime: string | number | Date): Promise<void>;
/**
 * [fs.promises] Asynchronously change file timestamps of the file referenced by the supplied `FileHandle`.
 * @param handle A `FileHandle`.
 * @param atime The last access time. If a string is provided, it will be coerced to number.
 * @param mtime The last modified time. If a string is provided, it will be coerced to number.
 */
export declare function futimes(handle: fs.promises.FileHandle, atime: string | number | Date, mtime: string | number | Date): Promise<void>;
/**
 * [fs.promises] Asynchronous realpath(3) - return the canonicalized absolute pathname.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * @param options The encoding (or an object specifying the encoding), used as the encoding of the result. If not provided, `'utf8'` is used.
 */
export declare function realpath(path: PathLike, options?: {
    encoding?: BufferEncoding | null;
} | BufferEncoding | null): Promise<string>;
export declare function realpath(path: PathLike, options: {
    encoding: 'buffer';
} | 'buffer'): Promise<Buffer>;
/**
 * [fs.promises] Asynchronously creates a unique temporary directory.
 * Generates six random characters to be appended behind a required `prefix` to create a unique temporary directory.
 * @param options The encoding (or an object specifying the encoding), used as the encoding of the result. If not provided, `'utf8'` is used.
 */
export declare function mkdtemp(prefix: string, options?: {
    encoding?: BufferEncoding | null;
} | BufferEncoding | null): Promise<string>;
export declare function mkdtemp(prefix: string, options: {
    encoding: 'buffer';
} | 'buffer'): Promise<Buffer>;
/**
 * [fs.promises] Asynchronously writes data to a file, replacing the file if it already exists.
 * It is unsafe to call `fsPromises.writeFile()` multiple times on the same file without waiting for the `Promise` to be resolved (or rejected).
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * URL support is _experimental_.
 * If a `FileHandle` is provided, the underlying file will _not_ be closed automatically.
 * @param data The data to write. If something other than a `Buffer` or `Uint8Array` is provided, the value is coerced to a string.
 * @param options Either the encoding for the file, or an object optionally specifying the encoding, file mode, and flag.
 * If `encoding` is not supplied, the default of `'utf8'` is used.
 * If `mode` is not supplied, the default of `0o666` is used.
 * If `mode` is a string, it is parsed as an octal integer.
 * If `flag` is not supplied, the default of `'w'` is used.
 */
export declare function writeFile(path: PathLike | fs.promises.FileHandle, data: any, options?: {
    encoding?: string | null;
    mode?: string | number;
    flag?: string | number;
} | string | null): Promise<void>;
/**
 * [fs.promises] Asynchronously append data to a file, creating the file if it does not exist.
 * @param file A path to a file. If a URL is provided, it must use the `file:` protocol.
 * URL support is _experimental_.
 * If a `FileHandle` is provided, the underlying file will _not_ be closed automatically.
 * @param data The data to write. If something other than a `Buffer` or `Uint8Array` is provided, the value is coerced to a string.
 * @param options Either the encoding for the file, or an object optionally specifying the encoding, file mode, and flag.
 * If `encoding` is not supplied, the default of `'utf8'` is used.
 * If `mode` is not supplied, the default of `0o666` is used.
 * If `mode` is a string, it is parsed as an octal integer.
 * If `flag` is not supplied, the default of `'a'` is used.
 */
export declare function appendFile(path: PathLike | fs.promises.FileHandle, data: any, options?: {
    encoding?: string | null;
    mode?: string | number;
    flag?: string | number;
} | string | null): Promise<void>;
/**
 * [fs.promises] Asynchronously reads the entire contents of a file.
 * @param path A path to a file. If a URL is provided, it must use the `file:` protocol.
 * If a `FileHandle` is provided, the underlying file will _not_ be closed automatically.
 * @param options An object that may contain an optional flag.
 * If a flag is not provided, it defaults to `'r'`.
 */
export declare function readFile(path: PathLike | fs.promises.FileHandle, options?: {
    encoding?: null;
    flag?: string | number;
} | null): Promise<Buffer>;
export declare function readFile(path: PathLike | fs.promises.FileHandle, options?: {
    encoding?: BufferEncoding;
    flag?: string | number;
} | BufferEncoding): Promise<string>;
