"use strict";
/**
 * Copyright (c) Facebook, Inc. and its affiliates.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.deploy = void 0;
const tslib_1 = require("tslib");
const fs_extra_1 = tslib_1.__importDefault(require("fs-extra"));
const path_1 = tslib_1.__importDefault(require("path"));
const os_1 = tslib_1.__importDefault(require("os"));
const logger_1 = tslib_1.__importDefault(require("@docusaurus/logger"));
const shelljs_1 = tslib_1.__importDefault(require("shelljs"));
const utils_1 = require("@docusaurus/utils");
const server_1 = require("../server");
const build_1 = require("./build");
// GIT_PASS env variable should not appear in logs
function obfuscateGitPass(str) {
    const gitPass = process.env.GIT_PASS;
    return gitPass ? str.replace(gitPass, 'GIT_PASS') : str;
}
// Log executed commands so that user can figure out mistakes on his own
// for example: https://github.com/facebook/docusaurus/issues/3875
function shellExecLog(cmd) {
    try {
        const result = shelljs_1.default.exec(cmd);
        logger_1.default.info `code=${obfuscateGitPass(cmd)} subdue=${`code: ${result.code}`}`;
        return result;
    }
    catch (err) {
        logger_1.default.error `code=${obfuscateGitPass(cmd)}`;
        throw err;
    }
}
async function deploy(siteDirParam = '.', cliOptions = {}) {
    const siteDir = await fs_extra_1.default.realpath(siteDirParam);
    const { outDir, siteConfig, siteConfigPath } = await (0, server_1.loadContext)({
        siteDir,
        config: cliOptions.config,
        outDir: cliOptions.outDir,
    });
    if (typeof siteConfig.trailingSlash === 'undefined') {
        logger_1.default.warn(`When deploying to GitHub Pages, it is better to use an explicit "trailingSlash" site config.
Otherwise, GitHub Pages will add an extra trailing slash to your site urls only on direct-access (not when navigation) with a server redirect.
This behavior can have SEO impacts and create relative link issues.
`);
    }
    logger_1.default.info('Deploy command invoked...');
    if (!shelljs_1.default.which('git')) {
        throw new Error('Git not installed or on the PATH!');
    }
    // Source repo is the repo from where the command is invoked
    const sourceRepoUrl = shelljs_1.default
        .exec('git remote get-url origin', { silent: true })
        .stdout.trim();
    // The source branch; defaults to the currently checked out branch
    const sourceBranch = process.env.CURRENT_BRANCH ??
        shelljs_1.default.exec('git rev-parse --abbrev-ref HEAD', { silent: true }).stdout.trim();
    const gitUser = process.env.GIT_USER;
    let useSSH = process.env.USE_SSH !== undefined &&
        process.env.USE_SSH.toLowerCase() === 'true';
    if (!gitUser && !useSSH) {
        // If USE_SSH is unspecified: try inferring from repo URL
        if (process.env.USE_SSH === undefined && (0, utils_1.hasSSHProtocol)(sourceRepoUrl)) {
            useSSH = true;
        }
        else {
            throw new Error('Please set the GIT_USER environment variable, or explicitly specify USE_SSH instead!');
        }
    }
    const organizationName = process.env.ORGANIZATION_NAME ??
        process.env.CIRCLE_PROJECT_USERNAME ??
        siteConfig.organizationName;
    if (!organizationName) {
        throw new Error(`Missing project organization name. Did you forget to define "organizationName" in ${siteConfigPath}? You may also export it via the ORGANIZATION_NAME environment variable.`);
    }
    logger_1.default.info `organizationName: name=${organizationName}`;
    const projectName = process.env.PROJECT_NAME ??
        process.env.CIRCLE_PROJECT_REPONAME ??
        siteConfig.projectName;
    if (!projectName) {
        throw new Error(`Missing project name. Did you forget to define "projectName" in ${siteConfigPath}? You may also export it via the PROJECT_NAME environment variable.`);
    }
    logger_1.default.info `projectName: name=${projectName}`;
    // We never deploy on pull request.
    const isPullRequest = process.env.CI_PULL_REQUEST ?? process.env.CIRCLE_PULL_REQUEST;
    if (isPullRequest) {
        shelljs_1.default.echo('Skipping deploy on a pull request.');
        shelljs_1.default.exit(0);
    }
    // github.io indicates organization repos that deploy via default branch. All
    // others use gh-pages (either case can be configured actually, but we can
    // make educated guesses). Organization deploys look like:
    // - Git repo: https://github.com/<organization>/<organization>.github.io
    // - Site url: https://<organization>.github.io
    const isGitHubPagesOrganizationDeploy = projectName.includes('.github.io');
    if (isGitHubPagesOrganizationDeploy &&
        !process.env.DEPLOYMENT_BRANCH &&
        !siteConfig.deploymentBranch) {
        throw new Error(`For GitHub pages organization deployments, 'docusaurus deploy' does not assume anymore that 'master' is your default Git branch.
Please provide the branch name to deploy to as an environment variable, for example DEPLOYMENT_BRANCH=main or DEPLOYMENT_BRANCH=master .
You can also set the deploymentBranch property in docusaurus.config.js .`);
    }
    const deploymentBranch = process.env.DEPLOYMENT_BRANCH ?? siteConfig.deploymentBranch ?? 'gh-pages';
    logger_1.default.info `deploymentBranch: name=${deploymentBranch}`;
    const githubHost = process.env.GITHUB_HOST ?? siteConfig.githubHost ?? 'github.com';
    const githubPort = process.env.GITHUB_PORT ?? siteConfig.githubPort;
    let deploymentRepoURL;
    if (useSSH) {
        deploymentRepoURL = (0, utils_1.buildSshUrl)(githubHost, organizationName, projectName, githubPort);
    }
    else {
        const gitPass = process.env.GIT_PASS;
        const gitCredentials = gitPass ? `${gitUser}:${gitPass}` : gitUser;
        deploymentRepoURL = (0, utils_1.buildHttpsUrl)(gitCredentials, githubHost, organizationName, projectName, githubPort);
    }
    logger_1.default.info `Remote repo URL: name=${obfuscateGitPass(deploymentRepoURL)}`;
    // Check if this is a cross-repo publish.
    const crossRepoPublish = !sourceRepoUrl.endsWith(`${organizationName}/${projectName}.git`);
    // We don't allow deploying to the same branch unless it's a cross publish.
    if (sourceBranch === deploymentBranch && !crossRepoPublish) {
        throw new Error(`You cannot deploy from this branch (${sourceBranch}).` +
            '\nYou will need to checkout to a different branch!');
    }
    // Save the commit hash that triggers publish-gh-pages before checking
    // out to deployment branch.
    const currentCommit = shellExecLog('git rev-parse HEAD').stdout.trim();
    const runDeploy = async (outputDirectory) => {
        const fromPath = outputDirectory;
        const toPath = await fs_extra_1.default.mkdtemp(path_1.default.join(os_1.default.tmpdir(), `${projectName}-${deploymentBranch}`));
        shelljs_1.default.cd(toPath);
        // Check out deployment branch when cloning repository, and then remove all
        // the files in the directory. If the 'clone' command fails, assume that
        // the deployment branch doesn't exist, and initialize git in an empty
        // directory, check out a clean deployment branch and add remote.
        if (shellExecLog(`git clone --depth 1 --branch ${deploymentBranch} ${deploymentRepoURL} "${toPath}"`).code === 0) {
            shellExecLog('git rm -rf .');
        }
        else {
            shellExecLog('git init');
            shellExecLog(`git checkout -b ${deploymentBranch}`);
            shellExecLog(`git remote add origin ${deploymentRepoURL}`);
        }
        try {
            await fs_extra_1.default.copy(fromPath, toPath);
        }
        catch (err) {
            logger_1.default.error `Copying build assets from path=${fromPath} to path=${toPath} failed.`;
            throw err;
        }
        shellExecLog('git add --all');
        const commitMessage = process.env.CUSTOM_COMMIT_MESSAGE ??
            `Deploy website - based on ${currentCommit}`;
        const commitResults = shellExecLog(`git commit -m "${commitMessage}"`);
        if (shellExecLog(`git push --force origin ${deploymentBranch}`).code !== 0) {
            throw new Error('Running "git push" command failed. Does the GitHub user account you are using have push access to the repository?');
        }
        else if (commitResults.code === 0) {
            // The commit might return a non-zero value when site is up to date.
            let websiteURL = '';
            if (githubHost === 'github.com') {
                websiteURL = projectName.includes('.github.io')
                    ? `https://${organizationName}.github.io/`
                    : `https://${organizationName}.github.io/${projectName}/`;
            }
            else {
                // GitHub enterprise hosting.
                websiteURL = `https://${githubHost}/pages/${organizationName}/${projectName}/`;
            }
            shelljs_1.default.echo(`Website is live at "${websiteURL}".`);
            shelljs_1.default.exit(0);
        }
    };
    if (!cliOptions.skipBuild) {
        // Build site, then push to deploymentBranch branch of specified repo.
        try {
            await (0, build_1.build)(siteDir, cliOptions, false).then(runDeploy);
        }
        catch (err) {
            logger_1.default.error('Deployment of the build output failed.');
            throw err;
        }
    }
    else {
        // Push current build to deploymentBranch branch of specified repo.
        await runDeploy(outDir);
    }
}
exports.deploy = deploy;
